<?php
ob_start();

// Enhanced session security
session_start();

// Security headers (PHP 5.6 compatible)
if (!headers_sent()) {
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    header('Referrer-Policy: strict-origin-when-cross-origin');
}

// Include necessary files
require_once(__DIR__.'/process/functions.php');
require_once(__DIR__.'/payment/stripe_config.php');

// Generate CSRF token if not exists (PHP 5.6 compatible)
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = md5(uniqid(mt_rand(), true));
}

// Security function to validate CSRF token
function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Enhanced input sanitization function
function sanitizeInput($input, $type = 'string') {
    if ($input === null) return '';
    
    $input = trim($input);
    
    switch ($type) {
        case 'email':
            return filter_var($input, FILTER_SANITIZE_EMAIL);
        case 'int':
            return filter_var($input, FILTER_SANITIZE_NUMBER_INT);
        case 'float':
            return filter_var($input, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        case 'username':
            // Allow only alphanumeric, underscore, hyphen
            return preg_replace('/[^a-zA-Z0-9_\-]/', '', $input);
        case 'phone':
            // Allow only numbers, spaces, parentheses, hyphens, plus sign
            return preg_replace('/[^0-9\s\(\)\-\+]/', '', $input);
        case 'name':
            // Allow only letters, spaces, apostrophes, hyphens
            return preg_replace('/[^a-zA-Z\s\'\-]/', '', $input);
        case 'address':
            // Allow alphanumeric, spaces, common punctuation for addresses
            return preg_replace('/[^a-zA-Z0-9\s\.\,\#\-]/', '', $input);
        default:
            // Default string sanitization
            return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    }
}

// Enhanced output escaping function
function escapeOutput($data) {
    if (is_array($data)) {
        return array_map('escapeOutput', $data);
    }
    return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
}

// Rate limiting function (simple implementation for PHP 5.6)
function checkRateLimit($action, $limit = 5, $window = 300) {
    $key = $action . '_' . $_SERVER['REMOTE_ADDR'];
    $current_time = time();
    
    if (!isset($_SESSION['rate_limits'])) {
        $_SESSION['rate_limits'] = array();
    }
    
    if (!isset($_SESSION['rate_limits'][$key])) {
        $_SESSION['rate_limits'][$key] = array('count' => 1, 'window_start' => $current_time);
        return true;
    }
    
    $rate_data = $_SESSION['rate_limits'][$key];
    
    // Reset window if enough time has passed
    if ($current_time - $rate_data['window_start'] > $window) {
        $_SESSION['rate_limits'][$key] = array('count' => 1, 'window_start' => $current_time);
        return true;
    }
    
    // Check if limit exceeded
    if ($rate_data['count'] >= $limit) {
        return false;
    }
    
    // Increment counter
    $_SESSION['rate_limits'][$key]['count']++;
    return true;
}

// Check if this is an AJAX request
function is_ajax() {
    return isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

// Handle AJAX requests
if (is_ajax()) {
    // CSRF protection for AJAX requests
    $csrf_token = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    if (!validateCSRFToken($csrf_token)) {
        http_response_code(403);
        echo json_encode(array('success' => false, 'message' => 'Invalid security token'));
        exit;
    }
    
    $action = sanitizeInput(isset($_POST['action']) ? $_POST['action'] : '', 'string');
    
    // Rate limiting for sensitive actions (temporarily disabled for create_account)
    $rate_limited_actions = array('send_otp', 'verify_otp');
    if (in_array($action, $rate_limited_actions)) {
        if (!checkRateLimit($action, 5, 300)) { // 5 attempts per 5 minutes
            http_response_code(429);
            echo json_encode(array('success' => false, 'message' => 'Too many attempts. Please try again later.'));
            exit;
        }
    }
    
    switch ($action) {
        case 'send_otp':
            handleSendOTP();
            break;
        case 'verify_otp':
            handleVerifyOTP();
            break;
        case 'check_username':
            handleCheckUsername();
            break;
        case 'check_email':
            handleCheckEmail();
            break;
        case 'check_phone':
            handleCheckPhone();
            break;
        case 'create_account':
            handleCreateAccount();
            break;
        default:
            http_response_code(400);
            echo json_encode(array('success' => false, 'message' => 'Invalid action'));
    }
    exit;
}

function handleSendOTP() {
    // Enhanced input validation and sanitization
    $phone = sanitizeInput(isset($_POST['phone']) ? $_POST['phone'] : '', 'phone');
    $country_code = sanitizeInput(isset($_POST['country_code']) ? $_POST['country_code'] : '+1', 'string');
    $email = sanitizeInput(isset($_POST['email']) ? $_POST['email'] : '', 'email');
    $method = sanitizeInput(isset($_POST['method']) ? $_POST['method'] : 'sms', 'string');
    
    // Validate input lengths
    if (strlen($phone) > 20 || strlen($email) > 100 || strlen($country_code) > 5) {
        echo json_encode(array('success' => false, 'message' => 'Invalid input length'));
        return;
    }
    
    // Validate method
    if (!in_array($method, array('sms', 'email'))) {
        echo json_encode(array('success' => false, 'message' => 'Invalid verification method'));
        return;
    }
    
    // Validate country code format
    if (!preg_match('/^\+\d{1,4}$/', $country_code)) {
        echo json_encode(array('success' => false, 'message' => 'Invalid country code format'));
        return;
    }
    
    if (empty($phone) && empty($email)) {
        echo json_encode(array('success' => false, 'message' => 'Phone number or email is required'));
        return;
    }
    
    // Validate email format if provided
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(array('success' => false, 'message' => 'Invalid email format'));
        return;
    }
    
    // Validate phone number format if provided
    if (!empty($phone) && !preg_match('/^\d{7,15}$/', $phone)) {
        echo json_encode(array('success' => false, 'message' => 'Invalid phone number format'));
        return;
    }
    
    // Combine country code with phone number for SMS
    $full_phone = $country_code . $phone;
    
    // Generate 6-digit OTP with secure random
    $otp = sprintf('%06d', mt_rand(100000, 999999));


    
    // Create a log file containing the OTP for debugging (remove in production)
    // This is for debugging purposes only, remove in production
    // file_put_contents('otp_log.txt', "OTP: {$otp} for Phone: {$full_phone}, Email: {$email}\n", FILE_APPEND);

    // Store OTP in session with enhanced security
    $_SESSION['otp'] = array(
        'code' => password_hash($otp, PASSWORD_BCRYPT), // Store hashed OTP
        'plain_code' => $otp, // Temporary for sending, will be unset
        'expires' => time() + 300, // 5 minutes
        'attempts' => 0,
        'max_attempts' => 3,
        'phone' => escapeOutput($full_phone),
        'email' => escapeOutput($email),
        'method' => escapeOutput($method),
        'ip' => $_SERVER['REMOTE_ADDR']
    );
    
    $success = false;
    $message = '';
    
    try {
        if ($method === 'sms' && !empty($phone)) {
            // Send OTP via SMS using full phone number with country code
            $sms_message = "Your DDxRx verification code is: {$otp}. This code will expire in 5 minutes. Do not share this code.";
            $result = send_sms($full_phone, $sms_message);
            
            if ($result) {
                $result_data = json_decode($result, true);
                $success = isset($result_data['result']) ? $result_data['result'] : false;
                $message = $success ? 'OTP sent successfully via SMS' : 'Failed to send SMS';
            }
        } elseif ($method === 'email' && !empty($email)) {
            // Send OTP via Email with enhanced security message
            $email_subject = 'DDxRx Account Verification Code';
            $email_message = "
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #f8f9fa; padding: 20px;'>
                <div style='background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);'>
                    <div style='text-align: center; margin-bottom: 30px;'>
                        <h1 style='color: #2c3e50; margin: 0; font-size: 28px;'>DDxRx</h1>
                        <p style='color: #7f8c8d; margin: 5px 0 0 0;'>Account Verification</p>
                    </div>
                    
                    <div style='text-align: center; margin-bottom: 30px;'>
                        <div style='background: #3498db; color: white; padding: 20px; border-radius: 10px; font-size: 36px; font-weight: bold; letter-spacing: 5px; margin: 20px 0;'>
                            {$otp}
                        </div>
                    </div>
                    
                    <div style='margin-bottom: 30px;'>
                        <p style='color: #2c3e50; margin: 0 0 15px 0; font-size: 16px; line-height: 1.5;'>
                            Use this verification code to complete your DDxRx account registration.
                        </p>
                        <p style='color: #e74c3c; margin: 0 0 15px 0; font-size: 14px; line-height: 1.5;'>
                            <strong>Security Notice:</strong> This code will expire in 5 minutes. Never share this code with anyone.
                        </p>
                    </div>
                    
                    <div style='background: #ecf0f1; padding: 15px; border-radius: 5px; margin-bottom: 20px;'>
                        <p style='color: #7f8c8d; margin: 0; font-size: 12px; line-height: 1.4;'>
                            If you didn't request this verification code, please ignore this email. 
                            Your account security has not been compromised.
                        </p>
                    </div>
                </div>
            </div>";
            
            // Use existing email function (assuming it exists)
            if (function_exists('send_email')) {
                $result = send_email($email, $email_subject, $email_message);
                $success = (bool) $result;
                $message = $success ? 'OTP sent successfully via Email' : 'Failed to send email';

            } else {
                $message = 'Email service not available';
            }
        }
        
        // Clear plain OTP from session for security
        if (isset($_SESSION['otp']['plain_code'])) {
            unset($_SESSION['otp']['plain_code']);
        }
        
    } catch (Exception $e) {
        $success = false;
        $message = 'Failed to send verification code';
        error_log('OTP Send Error: ' . $e->getMessage());
    }
    
    echo json_encode(array(
        'success' => $success, 
        'message' => escapeOutput($message),
        //'otp'=>$otp
    ));
}

function handleVerifyOTP() {
    // Enhanced input validation
    $entered_otp = sanitizeInput(isset($_POST['otp']) ? $_POST['otp'] : '', 'int');
    
    // Validate OTP format
    if (!preg_match('/^\d{6}$/', $entered_otp)) {
        echo json_encode(array('success' => false, 'message' => 'Invalid OTP format'));
        return;
    }
    
    if (!isset($_SESSION['otp'])) {
        echo json_encode(array('success' => false, 'message' => 'No OTP request found. Please request a new code.'));
        return;
    }
    
    $otp_data = $_SESSION['otp'];
    
    // Check for IP mismatch (additional security layer)
    if (isset($otp_data['ip']) && $otp_data['ip'] !== $_SERVER['REMOTE_ADDR']) {
        echo json_encode(array('success' => false, 'message' => 'Security validation failed'));
        return;
    }
    
    // Check if OTP has expired
    if (time() > $otp_data['expires']) {
        unset($_SESSION['otp']);
        echo json_encode(array('success' => false, 'message' => 'OTP has expired. Please request a new code.'));
        return;
    }
    
    // Check maximum attempts
    $max_attempts = isset($otp_data['max_attempts']) ? $otp_data['max_attempts'] : 3;
    if ($otp_data['attempts'] >= $max_attempts) {
        unset($_SESSION['otp']);
        echo json_encode(array('success' => false, 'message' => 'Too many failed attempts. Please request a new code.'));
        return;
    }
    
    // Verify OTP (using secure comparison)
    $otp_valid = false;
    if (isset($otp_data['code'])) {
        // If hashed OTP exists, verify against hash
        if (strlen($otp_data['code']) > 10) { // Likely a hash
            $otp_valid = password_verify($entered_otp, $otp_data['code']);
        } else {
            // Fallback for plain comparison (with constant-time comparison)
            $otp_valid = hash_equals($otp_data['code'], $entered_otp);
        }
    }
    
    if (!$otp_valid) {
        // Increment attempts counter
        $_SESSION['otp']['attempts']++;
        $remaining_attempts = $max_attempts - $_SESSION['otp']['attempts'];
        
        if ($remaining_attempts > 0) {
            echo json_encode(array(
                'success' => false, 
                'message' => "Invalid OTP. {$remaining_attempts} attempts remaining."
            ));
        } else {
            unset($_SESSION['otp']);
            echo json_encode(array(
                'success' => false, 
                'message' => 'Invalid OTP. Maximum attempts exceeded. Please request a new code.'
            ));
        }
        return;
    }
    
    // OTP verified successfully
    $_SESSION['otp_verified'] = true;
    $_SESSION['otp_verified_time'] = time();
    $_SESSION['verified_contact'] = array(
        'phone' => isset($otp_data['phone']) ? $otp_data['phone'] : '',
        'email' => isset($otp_data['email']) ? $otp_data['email'] : '',
        'method' => isset($otp_data['method']) ? $otp_data['method'] : ''
    );
    
    // Debug logging
    error_log('OTP Verification Successful - Session set: otp_verified=true, time=' . $_SESSION['otp_verified_time']);
    
    // Clear OTP data for security
    unset($_SESSION['otp']);
    
    echo json_encode(array(
        'success' => true, 
        'message' => 'OTP verified successfully'
    ));
}

function handleCheckUsername() {
    // Enhanced input validation and sanitization
    $username = sanitizeInput(isset($_POST['username']) ? $_POST['username'] : '', 'username');
    
    if (empty($username)) {
        echo json_encode(array('available' => false, 'message' => 'Username is required'));
        return;
    }
    
    // Validate username length and format
    if (strlen($username) < 3 || strlen($username) > 50) {
        echo json_encode(array('available' => false, 'message' => 'Username must be 3-50 characters long'));
        return;
    }
    
    if (!preg_match('/^[a-zA-Z0-9_\-]+$/', $username)) {
        echo json_encode(array('available' => false, 'message' => 'Username can only contain letters, numbers, underscore, and hyphen'));
        return;
    }
    
    try {
        // Check username availability using prepared statement
        $conn = ddxrxDBConnect();
        
        if (!$conn) {
            throw new Exception('Database connection failed');
        }
        
        $stmt = mysqli_prepare($conn, "SELECT userid FROM all_users WHERE username = ? LIMIT 1");
        if (!$stmt) {
            throw new Exception('Database prepare failed');
        }
        
        mysqli_stmt_bind_param($stmt, "s", $username);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        $available = (mysqli_num_rows($result) === 0);
        $message = $available ? 'Username is available' : 'Username is already taken';
        
        mysqli_stmt_close($stmt);
        mysqli_close($conn);
        
        echo json_encode(array(
            'available' => $available, 
            'message' => escapeOutput($message)
        ));
        
    } catch (Exception $e) {
        error_log('Username check error: ' . $e->getMessage());
        echo json_encode(array(
            'available' => false, 
            'message' => 'Unable to check username availability'
        ));
    }
}

function handleCheckEmail() {
    // Enhanced input validation and sanitization
    $email = sanitizeInput(isset($_POST['email']) ? $_POST['email'] : '', 'email');
    
    if (empty($email)) {
        echo json_encode(array('available' => false, 'message' => 'Email is required'));
        return;
    }
    
    // Validate email format and length
    if (strlen($email) > 100) {
        echo json_encode(array('available' => false, 'message' => 'Email is too long'));
        return;
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(array('available' => false, 'message' => 'Invalid email format'));
        return;
    }
    
    try {
        // Check email availability using prepared statement
        $conn = ddxrxDBConnect();
        
        if (!$conn) {
            throw new Exception('Database connection failed');
        }
        
        $stmt = mysqli_prepare($conn, "SELECT userid FROM all_users WHERE email_address = ? LIMIT 1");
        if (!$stmt) {
            throw new Exception('Database prepare failed');
        }
        
        mysqli_stmt_bind_param($stmt, "s", $email);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        $available = (mysqli_num_rows($result) === 0);
        $message = $available ? 'Email is available' : 'Email is already registered';
        
        mysqli_stmt_close($stmt);
        mysqli_close($conn);
        
        echo json_encode(array(
            'available' => $available, 
            'message' => escapeOutput($message)
        ));
        
    } catch (Exception $e) {
        error_log('Email check error: ' . $e->getMessage());
        echo json_encode(array(
            'available' => false, 
            'message' => 'Unable to check email availability'
        ));
    }
}

function handleCheckPhone() {
    // Enhanced input validation and sanitization
    $phone = sanitizeInput(isset($_POST['phone']) ? $_POST['phone'] : '', 'phone');
    $country_code = sanitizeInput(isset($_POST['country_code']) ? $_POST['country_code'] : '+1', 'string');
    
    if (empty($phone)) {
        echo json_encode(array('available' => false, 'message' => 'Phone number is required'));
        return;
    }
    
    // Validate country code format
    if (!preg_match('/^\+\d{1,4}$/', $country_code)) {
        echo json_encode(array('available' => false, 'message' => 'Invalid country code format'));
        return;
    }
    
    // Validate phone number format
    if (!preg_match('/^\d{7,15}$/', $phone)) {
        echo json_encode(array('available' => false, 'message' => 'Invalid phone number format'));
        return;
    }
    
    // Combine country code with phone number
    $full_phone = $country_code . $phone;
    
    // Validate combined phone length
    if (strlen($full_phone) > 20) {
        echo json_encode(array('available' => false, 'message' => 'Phone number is too long'));
        return;
    }
    
    try {
        // Check phone availability using prepared statement
        $conn = ddxrxDBConnect();
        
        if (!$conn) {
            throw new Exception('Database connection failed');
        }
        
        $stmt = mysqli_prepare($conn, "SELECT patient_id FROM patients WHERE cell = ? LIMIT 1");
        if (!$stmt) {
            throw new Exception('Database prepare failed');
        }
        
        mysqli_stmt_bind_param($stmt, "s", $full_phone);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        $available = (mysqli_num_rows($result) === 0);
        $message = $available ? 'Phone number is available' : 'Phone number is already registered';
        
        mysqli_stmt_close($stmt);
        mysqli_close($conn);
        
        echo json_encode(array(
            'available' => $available, 
            'message' => escapeOutput($message)
        ));
        
    } catch (Exception $e) {
        error_log('Phone check error: ' . $e->getMessage());
        echo json_encode(array(
            'available' => false, 
            'message' => 'Unable to check phone availability'
        ));
    }
}

function handleCreateAccount() {
    // Debug logging with timing
    $start_time = microtime(true);
    error_log('handleCreateAccount called at ' . date('Y-m-d H:i:s'));
    error_log('POST data keys: ' . implode(', ', array_keys($_POST)));
    
    // Enhanced security: Verify OTP was completed and within time limit
    if (!isset($_SESSION['otp_verified']) || !$_SESSION['otp_verified']) {
        // Debug logging
        error_log('OTP Verification Failed - Session state: ' . (isset($_SESSION['otp_verified']) ? 'set but false' : 'not set'));
        echo json_encode(array('success' => false, 'message' => 'Please verify your phone/email first'));
        return;
    }
    
    // Check OTP verification timestamp (valid for 30 minutes - increased from 10 minutes)
    if (isset($_SESSION['otp_verified_time']) && (time() - $_SESSION['otp_verified_time']) > 1800) {
        error_log('OTP Verification Expired - Time difference: ' . (time() - $_SESSION['otp_verified_time']) . ' seconds');
        unset($_SESSION['otp_verified'], $_SESSION['otp_verified_time']);
        echo json_encode(array('success' => false, 'message' => 'Verification expired. Please verify again.'));
        return;
    }
    
    // Enhanced input validation and sanitization
    $first_name = sanitizeInput(isset($_POST['first_name']) ? $_POST['first_name'] : '', 'name');
    $middle_name = sanitizeInput(isset($_POST['middle_name']) ? $_POST['middle_name'] : '', 'name');
    $last_name = sanitizeInput(isset($_POST['last_name']) ? $_POST['last_name'] : '', 'name');
    $birthday = sanitizeInput(isset($_POST['birthday']) ? $_POST['birthday'] : '', 'string');
    $gender = sanitizeInput(isset($_POST['gender']) ? $_POST['gender'] : '', 'string');
    $location = sanitizeInput(isset($_POST['location']) ? $_POST['location'] : '0', 'int');
    $phone = sanitizeInput(isset($_POST['phone']) ? $_POST['phone'] : '', 'phone');
    $country_code = sanitizeInput(isset($_POST['country_code']) ? $_POST['country_code'] : '+1', 'string');
    $email = sanitizeInput(isset($_POST['email']) ? $_POST['email'] : '', 'email');
    $username = sanitizeInput(isset($_POST['username']) ? $_POST['username'] : '', 'username');
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $billing_address = sanitizeInput(isset($_POST['billing_address']) ? $_POST['billing_address'] : '', 'address');
    $billing_city = sanitizeInput(isset($_POST['billing_city']) ? $_POST['billing_city'] : '', 'address');
    $billing_state = sanitizeInput(isset($_POST['billing_state']) ? $_POST['billing_state'] : '', 'address');
    $billing_zip = sanitizeInput(isset($_POST['billing_zip']) ? $_POST['billing_zip'] : '', 'address');
    $billing_country = sanitizeInput(isset($_POST['billing_country']) ? $_POST['billing_country'] : 'USA', 'address');
    $coupon_code = sanitizeInput(isset($_POST['coupon_code']) ? $_POST['coupon_code'] : '', 'string');
    $stripe_token = sanitizeInput(isset($_POST['stripeToken']) ? $_POST['stripeToken'] : '', 'string');
    $stripe_email = sanitizeInput(isset($_POST['stripeEmail']) ? $_POST['stripeEmail'] : '', 'email');
    $pricing = sanitizeInput(isset($_POST['pricing']) ? $_POST['pricing'] : '', 'string');
    
    // Enhanced validation with length limits
    $errors = array();
    
    // Name validation
    if (empty($first_name) || strlen($first_name) > 50) {
        $errors[] = 'First name is required and must be less than 50 characters';
    }
    if (strlen($middle_name) > 50) {
        $errors[] = 'Middle name must be less than 50 characters';
    }
    if (empty($last_name) || strlen($last_name) > 50) {
        $errors[] = 'Last name is required and must be less than 50 characters';
    }
    
    // Username validation
    if (empty($username) || strlen($username) < 3 || strlen($username) > 50) {
        $errors[] = 'Username is required and must be 3-50 characters';
    }
    if (!preg_match('/^[a-zA-Z0-9_\-]+$/', $username)) {
        $errors[] = 'Username can only contain letters, numbers, underscore, and hyphen';
    }
    
    // Email validation
    if (empty($email) || strlen($email) > 100) {
        $errors[] = 'Email is required and must be less than 100 characters';
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email format';
    }
    
    // Phone validation
    if (empty($phone) || !preg_match('/^\d{7,15}$/', $phone)) {
        $errors[] = 'Valid phone number is required';
    }
    if (!preg_match('/^\+\d{1,4}$/', $country_code)) {
        $errors[] = 'Invalid country code';
    }
    
    // Birthday validation
    if (empty($birthday) || $birthday === '--' || strpos($birthday, '-') === 0 || substr_count($birthday, '-') !== 2) {
        // Debug logging
        error_log('Birthday validation failed - birthday value: "' . $birthday . '"');
        $errors[] = 'Date of birth is required';
    } else {
        $date_parts = explode('-', $birthday);
        if (count($date_parts) !== 3 || empty($date_parts[0]) || empty($date_parts[1]) || empty($date_parts[2])) {
            // Debug logging
            error_log('Birthday parts validation failed - parts: ' . json_encode($date_parts));
            $errors[] = 'Please select a complete date of birth';
        } else {
            // Validate date format and ranges
            $year = (int)$date_parts[0];
            $month = (int)$date_parts[1];
            $day = (int)$date_parts[2];
            
            if ($year < 1900 || $year > date('Y') - 13) { // Minimum age 13
                $errors[] = 'Invalid birth year';
            }
            if ($month < 1 || $month > 12) {
                $errors[] = 'Invalid birth month';
            }
            if ($day < 1 || $day > 31) {
                $errors[] = 'Invalid birth day';
            }
            if (!checkdate($month, $day, $year)) {
                $errors[] = 'Invalid date of birth';
            }
        }
    }
    
    // Gender validation
    if (empty($gender) || !in_array($gender, array('m', 'f', 'o'))) {
        $errors[] = 'Valid gender is required';
    }
    
    // Password validation
    if (empty($password)) {
        $errors[] = 'Password is required';
    } else {
        if (strlen($password) < 8) {
            $errors[] = 'Password must be at least 8 characters';
        }
        if (strlen($password) > 128) {
            $errors[] = 'Password is too long';
        }
        // Check for common weak passwords
        $weak_passwords = array('password', '123456', 'qwerty', 'admin', 'letmein');
        if (in_array(strtolower($password), $weak_passwords)) {
            $errors[] = 'Password is too weak';
        }
    }
    
    // Billing address validation (if provided)
    if (!empty($billing_address) && strlen($billing_address) > 255) {
        $errors[] = 'Billing address is too long';
    }
    if (!empty($billing_city) && strlen($billing_city) > 100) {
        $errors[] = 'Billing city is too long';
    }
    if (!empty($billing_state) && strlen($billing_state) > 100) {
        $errors[] = 'Billing state is too long';
    }
    if (!empty($billing_zip) && strlen($billing_zip) > 20) {
        $errors[] = 'Billing zip code is too long';
    }
    
    if (!empty($errors)) {
        echo json_encode(array('success' => false, 'message' => implode(', ', $errors)));
        return;
    }
    
    
    try {
        // First, connect to database for validation checks only
        $conn = ddxrxDBConnect();
        
        if (!$conn) {
            throw new Exception('Database connection failed');
        }
        
        error_log('Database connected for validation - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
        
        // Check username availability
        $username_check = "SELECT userid FROM all_users WHERE username = '" . mysqli_real_escape_string($conn, $username) . "' LIMIT 1";
        $result = mysqli_query($conn, $username_check);
        
        if (mysqli_num_rows($result) > 0) {
            mysqli_close($conn);
            echo json_encode(array('success' => false, 'message' => 'Username is already taken'));
            return;
        }
        
        // Check email availability
        $email_check = "SELECT userid FROM all_users WHERE email_address = '" . mysqli_real_escape_string($conn, $email) . "' LIMIT 1";
        $result = mysqli_query($conn, $email_check);
        
        if (mysqli_num_rows($result) > 0) {
            mysqli_close($conn);
            echo json_encode(array('success' => false, 'message' => 'Email is already registered'));
            return;
        }
        
        // Check phone availability
        $full_phone = $country_code . $phone;
        $phone_check = "SELECT patient_id FROM patients WHERE cell = '" . mysqli_real_escape_string($conn, $full_phone) . "' LIMIT 1";
        $result = mysqli_query($conn, $phone_check);
        
        if (mysqli_num_rows($result) > 0) {
            mysqli_close($conn);
            echo json_encode(array('success' => false, 'message' => 'Phone number is already registered'));
            return;
        }
        
        // Close connection after validation
        mysqli_close($conn);
        
        error_log('Validation checks completed - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
        
        // Prepare data for account creation (but don't create accounts yet)
        $password_hash = password_hash($password, PASSWORD_BCRYPT, array('cost' => 12));
        $user_id = randomIDGenerator($username);
        $patient_id = randomIDGenerator($user_id);
        $epic_id = md5(uniqid(mt_rand(), true));
        $occupation = 'Not Specified';
        $billing_company = '';
        $billing_name = $first_name . ' ' . $last_name;
        $mrn_hospital = 'Not Specified';
        $encounter_id = 'Not Specified';
        

        // Handle payment processing FIRST - only create accounts if payment succeeds
        $coupon_applied = false;
        $payment_processed = false;
        
        /*if (!empty($coupon_code)) {
            // Debug logging
            error_log('Processing coupon code: ' . $coupon_code);
            
            // Validate coupon code format
            if (!preg_match('/^[a-zA-Z0-9]+$/', $coupon_code)) {
                throw new Exception('Invalid coupon code format');
            }
            
            // Load coupon definitions
            $coupons = unserialize(COUPONS);
            error_log('Available coupons: ' . json_encode($coupons));
            
            if (array_key_exists($coupon_code, $coupons)) {
                error_log('Coupon found: ' . $coupon_code . ' = ' . $coupons[$coupon_code]);
                $coupon_applied = true;
                $payment_processed = true;
                error_log('Coupon validation successful - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
            } else {
                error_log('Invalid coupon code provided: ' . $coupon_code);
                throw new Exception('Invalid coupon code');
            }
        } elseif (!empty($stripe_token) && !empty($stripe_email) && !empty($pricing)) {
            // Debug logging
            error_log('Processing Stripe payment with token: ' . substr($stripe_token, 0, 10) . '...');
            
            // Enhanced security: Validate Stripe inputs
            if (!filter_var($stripe_email, FILTER_VALIDATE_EMAIL)) {
                throw new Exception('Invalid Stripe email format');
            }
            
            if (!preg_match('/^[a-zA-Z0-9_]+$/', $pricing)) {
                throw new Exception('Invalid pricing plan format');
            }
            
            // Handle Stripe payment with enhanced error handling
            try {
                require_once(__DIR__.'/payment/stripe/init.php');
                
                // Set Stripe API key
                if (defined('STRIPE_MODE') && STRIPE_MODE == 'live') {
                    \Stripe\Stripe::setApiKey(STRIPE_LIVE_SECRET_KEY);
                    $plans = unserialize(STRIPE_LIVE_PLAN);
                    $subscription_expired_days = unserialize(SUBSCRIPTION_EXPIRED_DAYS);
                    error_log('Using LIVE Stripe mode');
                } else {
                    \Stripe\Stripe::setApiKey(STRIPE_TEST_SECRET_KEY);
                    $plans = unserialize(STRIPE_TEST_PLAN);
                    $subscription_expired_days = unserialize(SUBSCRIPTION_EXPIRED_DAYS);
                    error_log('Using TEST Stripe mode');
                }
                
                // Validate pricing plan exists
                if (!isset($plans[$pricing])) {
                    error_log('Invalid pricing plan: ' . $pricing . '. Available plans: ' . json_encode(array_keys($plans)));
                    throw new Exception('Invalid pricing plan');
                }
                
                // Calculate expiration date based on plan
                if (isset($subscription_expired_days[$pricing])) {
                    $expired_days = (int) $subscription_expired_days[$pricing];
                    $expired_date = date('Y-m-d', strtotime("+ $expired_days days"));
                } else {
                    $expired_date = date('Y-m-d', strtotime('+30 days')); // Default 30 days
                }
                
                error_log('Creating Stripe customer for email: ' . $stripe_email);
                
                // Create Stripe customer with error handling
                $customer = \Stripe\Customer::create(array(
                    'email' => $stripe_email,
                    'source' => $stripe_token,
                    'description' => 'DDxRx Patient: ' . $first_name . ' ' . $last_name
                ));
                
                error_log('Stripe customer created: ' . $customer->id);
                
                // Create subscription with error handling
                $subscription = \Stripe\Subscription::create(array(
                    'customer' => $customer->id,
                    'items' => array(array('plan' => $plans[$pricing])),
                    'metadata' => array(
                        'patient_id' => $patient_id,
                        'username' => $username
                    )
                ));
                
                error_log('Stripe subscription created: ' . $subscription->id . ' with status: ' . $subscription->status);
                
                if ($subscription->status !== 'incomplete') {
                    $payment_processed = true;
                    // Store Stripe data for later database insertion
                    $stripe_data = array(
                        'customer' => $customer,
                        'subscription' => $subscription,
                        'expired_date' => $expired_date,
                        'pricing' => $pricing
                    );
                    error_log('Stripe payment successful - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
                } else {
                    throw new Exception('Payment was not completed successfully');
                }
                
            } catch (\Stripe\Exception\CardException $e) {
                // Card was declined
                throw new Exception('Payment failed: ' . $e->getError()->message);
            } catch (\Stripe\Exception\RateLimitException $e) {
                // Too many requests to Stripe API
                throw new Exception('Payment service temporarily unavailable');
            } catch (\Stripe\Exception\InvalidRequestException $e) {
                // Invalid parameters
                throw new Exception('Payment processing error');
            } catch (\Stripe\Exception\AuthenticationException $e) {
                // Authentication failed
                error_log('Stripe authentication failed: ' . $e->getMessage());
                throw new Exception('Payment service configuration error');
            } catch (\Stripe\Exception\ApiConnectionException $e) {
                // Network communication failed
                throw new Exception('Payment service connection failed');
            } catch (\Stripe\Exception\ApiErrorException $e) {
                // Generic API error
                error_log('Stripe API error: ' . $e->getMessage());
                throw new Exception('Payment processing failed');
            } catch (Exception $stripe_error) {
                // Other errors
                error_log('Payment error: ' . $stripe_error->getMessage());
                throw new Exception('Payment processing failed');
            }
        }
        
        // Only create database records if payment was successful
        if (!$payment_processed) {
            throw new Exception('Payment is required to create account');
        }
        
        error_log('Payment successful, creating database records - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
        */


        // Now create the database records since payment succeeded
        $conn = ddxrxDBConnect();
        
        if (!$conn) {
            throw new Exception('Database connection failed after payment processing');
        }
        
        // Insert into all_users table with proper escaping
        $user_query = "INSERT INTO all_users (
            userid, groupid, username, password_new, email_address, user_type, location_id, is_active
        ) VALUES (
            '" . mysqli_real_escape_string($conn, $user_id) . "',
            '100001',
            '" . mysqli_real_escape_string($conn, $username) . "',
            '" . mysqli_real_escape_string($conn, $password_hash) . "',
            '" . mysqli_real_escape_string($conn, $email) . "',
            '3',
            '1',
            '1'
        )";
        
        if (!mysqli_query($conn, $user_query)) {
            throw new Exception('Failed to create user account: ' . mysqli_error($conn));
        }
        
        error_log('User account created - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
        
        // Get the actual inserted user ID
        $actual_user_id = mysqli_insert_id($conn);
        if (!$actual_user_id) {
            $actual_user_id = $user_id; // fallback to generated ID
        }
        
        // Insert into patients table with proper escaping
        $patient_query = "INSERT INTO patients (
            patient_id, groupid, location_id, user_id, patient_fname, mname, patient_lname, 
            patient_dob, patient_gender, epic_id, active, cell, email_id, occupation, 
            billing_company, billing_name, billing_address, billing_city, billing_state, 
            billing_zip, billing_country, billing_email, mrn_hospital, encounter_id
        ) VALUES (
            '" . mysqli_real_escape_string($conn, $patient_id) . "',
            '100001',
            '1', 
            '" . mysqli_real_escape_string($conn, $actual_user_id) . "',
            '" . mysqli_real_escape_string($conn, $first_name) . "',
            '" . mysqli_real_escape_string($conn, $middle_name) . "',
            '" . mysqli_real_escape_string($conn, $last_name) . "',
            '" . mysqli_real_escape_string($conn, $birthday) . "',
            '" . mysqli_real_escape_string($conn, $gender) . "',
            '" . mysqli_real_escape_string($conn, $epic_id) . "',
            '1',
            '" . mysqli_real_escape_string($conn, $full_phone) . "',
            '" . mysqli_real_escape_string($conn, $email) . "',
            '" . mysqli_real_escape_string($conn, $occupation) . "',
            '" . mysqli_real_escape_string($conn, $billing_company) . "',
            '" . mysqli_real_escape_string($conn, $billing_name) . "',
            '" . mysqli_real_escape_string($conn, $billing_address) . "',
            '" . mysqli_real_escape_string($conn, $billing_city) . "',
            '" . mysqli_real_escape_string($conn, $billing_state) . "',
            '" . mysqli_real_escape_string($conn, $billing_zip) . "',
            '" . mysqli_real_escape_string($conn, $billing_country) . "',
            '" . mysqli_real_escape_string($conn, $email) . "',
            '" . mysqli_real_escape_string($conn, $mrn_hospital) . "',
            '" . mysqli_real_escape_string($conn, $encounter_id) . "'
        )";
        
        if (!mysqli_query($conn, $patient_query)) {
            throw new Exception('Failed to create patient record: ' . mysqli_error($conn));
        }
        
        error_log('Patient record created - time elapsed: ' . round((microtime(true) - $start_time) * 1000, 2) . 'ms');
        
        // Now update patient record with payment information
        if ($coupon_applied && !empty($coupon_code)) {
            $coupons = unserialize(COUPONS);
            // Handle different coupon types
            if ($coupons[$coupon_code] === 'lifetime') {
                error_log('Applying lifetime coupon for patient_id: ' . $patient_id);
                // Lifetime access - set expiration far into future
                $update_query = "UPDATE patients SET 
                    stripe_plan_type = 'lifetime', 
                    stripe_subscription_isactive = 1, 
                    subscription_expired_date = '2099-12-31' 
                    WHERE patient_id = '" . mysqli_real_escape_string($conn, $patient_id) . "'";
            } else {
                // Trial period in days
                $trial_days = (int) $coupons[$coupon_code];
                $expired_date = date('Y-m-d', strtotime("+ $trial_days days"));
                error_log('Applying trial coupon for ' . $trial_days . ' days, expiring: ' . $expired_date);
                
                $update_query = "UPDATE patients SET 
                    stripe_plan_type = 'trial', 
                    stripe_subscription_isactive = 1, 
                    subscription_expired_date = '" . mysqli_real_escape_string($conn, $expired_date) . "' 
                    WHERE patient_id = '" . mysqli_real_escape_string($conn, $patient_id) . "'";
            }
            
            if (!mysqli_query($conn, $update_query)) {
                error_log('Coupon update failed: ' . mysqli_error($conn));
                throw new Exception('Failed to apply coupon: ' . mysqli_error($conn));
            }
            
            error_log('Coupon applied successfully');
        } elseif (isset($stripe_data)) {
            // Update with Stripe information
            $customer_json = json_encode($stripe_data['customer']);
            $subscription_json = json_encode($stripe_data['subscription']);
            
            $stripe_update_query = "UPDATE patients SET 
                stripe_plan_type = '" . mysqli_real_escape_string($conn, $stripe_data['pricing']) . "',
                subscription_expired_date = '" . mysqli_real_escape_string($conn, $stripe_data['expired_date']) . "',
                stripe_customer_id = '" . mysqli_real_escape_string($conn, $stripe_data['customer']->id) . "',
                stripe_customer_body = '" . mysqli_real_escape_string($conn, $customer_json) . "',
                stripe_subscription_id = '" . mysqli_real_escape_string($conn, $stripe_data['subscription']->id) . "',
                stripe_subscription_body = '" . mysqli_real_escape_string($conn, $subscription_json) . "',
                stripe_subscription_isactive = 1
                WHERE patient_id = '" . mysqli_real_escape_string($conn, $patient_id) . "'";
            
            if (!mysqli_query($conn, $stripe_update_query)) {
                throw new Exception('Failed to update payment information: ' . mysqli_error($conn));
            }
            
            error_log('Stripe payment information updated successfully');
        }
        
        // Store user data in session with security measures
        $_SESSION['new_user'] = array(
            'user_id' => escapeOutput($user_id),
            'patient_id' => escapeOutput($patient_id),
            'username' => escapeOutput($username),
            'email' => escapeOutput($email),
            'name' => escapeOutput($first_name . ' ' . $last_name),
            'created_time' => time()
        );
        
        // Close database connection and mark it as closed
        mysqli_close($conn);
        $conn = null; // Clear the connection variable to prevent accidental reuse
        
        // Clear OTP verification for security (only after successful completion)
        unset($_SESSION['otp_verified'], $_SESSION['otp_verified_time']);
        
        $total_time = round((microtime(true) - $start_time) * 1000, 2);
        error_log('Account creation completed successfully - total time: ' . $total_time . 'ms');
        
        echo json_encode(array(
            'success' => true, 
            'message' => escapeOutput('Account created successfully'),
            'redirect' => true,
            'coupon_applied' => $coupon_applied,
            'payment_processed' => $payment_processed,
            'processing_time_ms' => $total_time
        ));
        
    } catch (Exception $e) {
        // Enhanced error handling - ensure no database records are left if payment failed
        // Only attempt cleanup if we have a valid, open connection
        if (isset($conn) && $conn instanceof mysqli) {
            // Use a more robust method to check if connection is still valid
            // We'll use mysqli_stat() which returns false for closed connections
            $connection_active = false;
            
            // Suppress all errors and warnings when checking connection status
            $old_error_level = error_reporting(0);
            set_error_handler(function() { /* ignore all errors */ });
            
            try {
                // mysqli_stat() will return false if connection is closed
                $stat_result = @mysqli_stat($conn);
                if ($stat_result !== false && $stat_result !== null) {
                    $connection_active = true;
                }
            } catch (Throwable $connection_error) {
                // Connection is closed or invalid, no cleanup needed
                $connection_active = false;
            } finally {
                // Restore original error handling
                restore_error_handler();
                error_reporting($old_error_level);
            }
            
            if ($connection_active) {
                // If we reach this point and have an active connection, payment failed
                // Clean up any potential partial records (defensive programming)
                if (isset($user_id)) {
                    $escaped_user_id = @mysqli_real_escape_string($conn, $user_id);
                    @mysqli_query($conn, "DELETE FROM all_users WHERE userid = '" . $escaped_user_id . "'");
                }
                if (isset($patient_id)) {
                    $escaped_patient_id = @mysqli_real_escape_string($conn, $patient_id);
                    @mysqli_query($conn, "DELETE FROM patients WHERE patient_id = '" . $escaped_patient_id . "'");
                }
                @mysqli_close($conn);
            }
            // If connection was already closed, no cleanup needed
        }
        
        $total_time = round((microtime(true) - $start_time) * 1000, 2);
        
        // Log error for debugging (without sensitive data)
        error_log('Account creation error after ' . $total_time . 'ms: ' . $e->getMessage());
        
        // Return more specific error message for debugging
        $error_msg = $e->getMessage();
        
        echo json_encode(array(
            'success' => false, 
            'message' => escapeOutput($error_msg),
            'debug_info' => 'Check server logs for more details',
            'processing_time_ms' => $total_time
        ));
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sign Up - DDxRx AI Healthcare Platform</title>
    
    <!-- Existing stylesheets -->
    <link rel="stylesheet" type="text/css" href="assets/css/reset.css<?php echo getAssetVersion(); ?>">
    <link rel="stylesheet" type="text/css" href="assets/css/grid.css<?php echo getAssetVersion(); ?>">
    
    <!-- New custom stylesheet -->
    <link rel="stylesheet" type="text/css" href="assets/css/new-signup-patient.css<?php echo getAssetVersion(); ?>">
    
    <!-- Scripts -->
    <script type="text/javascript" src="assets/js/jquery.min.js<?php echo getAssetVersion(); ?>"></script>
    <script src="//js.stripe.com/v3/"></script>
    
    <?php if(defined('STRIPE_MODE')): ?>
    <script type="text/javascript">
        const stripe_mode = "<?php echo escapeOutput(STRIPE_MODE); ?>";
        <?php if(STRIPE_MODE !='live'): ?>
            const publishable_key = "<?php echo escapeOutput(STRIPE_TEST_PUBLISHABLE_KEY); ?>";
        <?php else: ?>
            const publishable_key = "<?php echo escapeOutput(STRIPE_LIVE_PUBLISHABLE_KEY); ?>";
        <?php endif ?>
        // CSRF token for AJAX requests
        const csrf_token = "<?php echo escapeOutput($_SESSION['csrf_token']); ?>";
    </script>
    <?php endif ?>
    
    <!-- SweetAlert2 -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

</head>
<body>
    <div class="new-signup-wrapper">
        <div class="new-signup-container">
            <!-- Header -->
            <div class="new-signup-header">
                <h1>Create Your Account</h1>
                <p class="subtitle">Join the future of AI-powered healthcare</p>
            </div>
            
            <!-- Progress Indicator -->
            <div class="progress-container">
                <div class="progress-steps">
                    <div class="progress-line"></div>
                    <div class="step active" data-step="1">1</div>
                    <div class="step" data-step="2">2</div>
                    <div class="step" data-step="3">3</div>
                    <div class="step" data-step="4">4</div>
                    <!--<div class="step" data-step="5">5</div>-->
                </div>
            </div>
            
            <!-- Form Content -->
            <div class="form-content">
                <!-- Step 1: Personal Information -->
                <div class="section active" id="step-1">
                    <h2>Personal Information</h2>
                    <form id="personal-info-form">
                        <div class="form-group">
                            <label for="first_name">First Name <span class="required">*</span></label>
                            <input type="text" id="first_name" name="first_name" class="form-control" required maxlength="15">
                            <span class="field-error" id="first_name_error"></span>
                        </div>
                        
                        <div class="form-group">
                            <label for="middle_name">Middle Name</label>
                            <input type="text" id="middle_name" name="middle_name" class="form-control" maxlength="15">
                        </div>
                        
                        <div class="form-group">
                            <label for="last_name">Last Name <span class="required">*</span></label>
                            <input type="text" id="last_name" name="last_name" class="form-control" required maxlength="15">
                            <span class="field-error" id="last_name_error"></span>
                        </div>
                        
                        <div class="btn-container">
                            <button type="button" class="btn btn-primary" id="next-step-1">Next</button>
                        </div>
                    </form>
                </div>
                
                <!-- Step 2: Date of Birth & Gender -->
                <div class="section" id="step-2">
                    <h2>Date of Birth & Gender</h2>
                    <form id="birth-gender-form">
                        <div class="form-group">
                            <label>Date of Birth <span class="required">*</span></label>
                            <div class="form-row">
                                <div class="form-group">
                                    <select id="birth_month" name="birth_month" class="form-control" required>
                                        <option value="">Month</option>
                                        <?php for ($x = 1; $x <= 12; $x++): ?>
                                            <option value="<?php echo sprintf('%02d', $x); ?>"><?php echo sprintf('%02d', $x); ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <select id="birth_day" name="birth_day" class="form-control" required>
                                        <option value="">Day</option>
                                        <?php for ($x = 1; $x <= 31; $x++): ?>
                                            <option value="<?php echo sprintf('%02d', $x); ?>"><?php echo sprintf('%02d', $x); ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <select id="birth_year" name="birth_year" class="form-control" required>
                                        <option value="">Year</option>
                                        <?php for ($x = 1900; $x <= date("Y"); $x++): ?>
                                            <option value="<?php echo $x; ?>"><?php echo $x; ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                            </div>
                            <span class="field-error" id="birthday_error"></span>
                        </div>
                        
                        <div class="form-group">
                            <label>Gender <span class="required">*</span></label>
                            <div class="radio-group">
                                <div class="radio-item selected" data-value="m">
                                    <input type="radio" name="gender" value="m" id="gender_m" checked>
                                    <label for="gender_m">Male</label>
                                </div>
                                <div class="radio-item" data-value="f">
                                    <input type="radio" name="gender" value="f" id="gender_f">
                                    <label for="gender_f">Female</label>
                                </div>
                                <div class="radio-item" data-value="o">
                                    <input type="radio" name="gender" value="o" id="gender_o">
                                    <label for="gender_o">Other</label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="btn-container">
                            <button type="button" class="btn btn-secondary" id="prev-step-2">Previous</button>
                            <button type="button" class="btn btn-primary" id="next-step-2">Next</button>
                        </div>
                    </form>
                </div>
                
                <!-- Step 3: Contact Information -->
                <div class="section" id="step-3">
                    <h2>Contact Information</h2>
                    <form id="contact-form">
                        <div class="form-group">
                            <label>Patient Location</label>
                            <div class="radio-group">
                                <div class="radio-item selected" data-value="0">
                                    <input type="radio" name="location" value="0" id="location_home" checked>
                                    <label for="location_home">Home</label>
                                </div>
                                <div class="radio-item" data-value="1">
                                    <input type="radio" name="location" value="1" id="location_hospital">
                                    <label for="location_hospital">Hospital & Clinic</label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="phone">Phone Number <span class="required">*</span></label>
                            <div class="phone-input-group">
                                <select id="country_code" name="country_code" class="form-control country-select" required>
                                    <option value="+1" selected>🇺🇸 +1 (US)</option>
                                    <option value="+44">🇬🇧 +44 (UK)</option>
                                    <option value="+1">🇨🇦 +1 (CA)</option>
                                    <option value="+61">🇦🇺 +61 (AU)</option>
                                    <option value="+33">🇫🇷 +33 (FR)</option>
                                    <option value="+49">🇩🇪 +49 (DE)</option>
                                    <option value="+39">🇮🇹 +39 (IT)</option>
                                    <option value="+34">🇪🇸 +34 (ES)</option>
                                    <option value="+31">🇳🇱 +31 (NL)</option>
                                    <option value="+41">🇨🇭 +41 (CH)</option>
                                    <option value="+43">🇦🇹 +43 (AT)</option>
                                    <option value="+32">🇧🇪 +32 (BE)</option>
                                    <option value="+45">🇩🇰 +45 (DK)</option>
                                    <option value="+358">🇫🇮 +358 (FI)</option>
                                    <option value="+47">🇳🇴 +47 (NO)</option>
                                    <option value="+46">🇸🇪 +46 (SE)</option>
                                    <option value="+48">🇵🇱 +48 (PL)</option>
                                    <option value="+420">🇨🇿 +420 (CZ)</option>
                                    <option value="+36">🇭🇺 +36 (HU)</option>
                                    <option value="+351">🇵🇹 +351 (PT)</option>
                                    <option value="+30">🇬🇷 +30 (GR)</option>
                                    <option value="+353">🇮🇪 +353 (IE)</option>
                                    <option value="+7">🇷🇺 +7 (RU)</option>
                                    <option value="+380">🇺🇦 +380 (UA)</option>
                                    <option value="+90">🇹🇷 +90 (TR)</option>
                                    <option value="+972">🇮🇱 +972 (IL)</option>
                                    <option value="+971">🇦🇪 +971 (AE)</option>
                                    <option value="+966">🇸🇦 +966 (SA)</option>
                                    <option value="+974">🇶🇦 +974 (QA)</option>
                                    <option value="+965">🇰🇼 +965 (KW)</option>
                                    <option value="+968">🇴🇲 +968 (OM)</option>
                                    <option value="+973">🇧🇭 +973 (BH)</option>
                                    <option value="+20">🇪🇬 +20 (EG)</option>
                                    <option value="+27">🇿🇦 +27 (ZA)</option>
                                    <option value="+234">🇳🇬 +234 (NG)</option>
                                    <option value="+254">🇰🇪 +254 (KE)</option>
                                    <option value="+81">🇯🇵 +81 (JP)</option>
                                    <option value="+82">🇰🇷 +82 (KR)</option>
                                    <option value="+86">🇨🇳 +86 (CN)</option>
                                    <option value="+852">🇭🇰 +852 (HK)</option>
                                    <option value="+853">🇲🇴 +853 (MO)</option>
                                    <option value="+886">🇹🇼 +886 (TW)</option>
                                    <option value="+91">🇮🇳 +91 (IN)</option>
                                    <option value="+92">🇵🇰 +92 (PK)</option>
                                    <option value="+880">🇧🇩 +880 (BD)</option>
                                    <option value="+94">🇱🇰 +94 (LK)</option>
                                    <option value="+977">🇳🇵 +977 (NP)</option>
                                    <option value="+63">🇵🇭 +63 (PH)</option>
                                    <option value="+65">🇸🇬 +65 (SG)</option>
                                    <option value="+60">🇲🇾 +60 (MY)</option>
                                    <option value="+66">🇹🇭 +66 (TH)</option>
                                    <option value="+84">🇻🇳 +84 (VN)</option>
                                    <option value="+62">🇮🇩 +62 (ID)</option>
                                    <option value="+95">🇲🇲 +95 (MM)</option>
                                    <option value="+855">🇰🇭 +855 (KH)</option>
                                    <option value="+856">🇱🇦 +856 (LA)</option>
                                    <option value="+673">🇧🇳 +673 (BN)</option>
                                    <option value="+670">🇹🇱 +670 (TL)</option>
                                    <option value="+52">🇲🇽 +52 (MX)</option>
                                    <option value="+54">🇦🇷 +54 (AR)</option>
                                    <option value="+55">🇧🇷 +55 (BR)</option>
                                    <option value="+56">🇨🇱 +56 (CL)</option>
                                    <option value="+57">🇨🇴 +57 (CO)</option>
                                    <option value="+58">🇻🇪 +58 (VE)</option>
                                    <option value="+51">🇵🇪 +51 (PE)</option>
                                    <option value="+593">🇪🇨 +593 (EC)</option>
                                    <option value="+591">🇧🇴 +591 (BO)</option>
                                    <option value="+595">🇵🇾 +595 (PY)</option>
                                    <option value="+598">🇺🇾 +598 (UY)</option>
                                    <option value="+507">🇵🇦 +507 (PA)</option>
                                    <option value="+506">🇨🇷 +506 (CR)</option>
                                    <option value="+502">🇬🇹 +502 (GT)</option>
                                    <option value="+503">🇸🇻 +503 (SV)</option>
                                    <option value="+504">🇭🇳 +504 (HN)</option>
                                    <option value="+505">🇳🇮 +505 (NI)</option>
                                    <option value="+1">🇯🇲 +1 (JM)</option>
                                    <option value="+1">🇹🇹 +1 (TT)</option>
                                    <option value="+1">🇧🇧 +1 (BB)</option>
                                    <option value="+1">🇧🇸 +1 (BS)</option>
                                </select>
                                <input type="tel" id="phone" name="phone" class="form-control phone-number" required maxlength="15" placeholder="Enter phone number">
                            </div>
                            <span class="field-error" id="phone_error"></span>
                            <span class="field-success" id="phone_success"></span>
                        </div>
                        
                        <div class="form-group">
                            <label for="email">Email Address <span class="required">*</span></label>
                            <input type="email" id="email" name="email" class="form-control" required>
                            <span class="field-error" id="email_error"></span>
                            <span class="field-success" id="email_success"></span>
                        </div>
                        
                        <div class="btn-container">
                            <button type="button" class="btn btn-secondary" id="prev-step-3">Previous</button>
                            <button type="button" class="btn btn-primary" id="next-step-3">Next</button>
                        </div>
                    </form>
                </div>
                
                <!-- Step 4: Account Details & OTP Verification -->
                <div class="section" id="step-4">
                    <h2>Account Details</h2>
                    <form id="account-form">
                        <div class="form-group">
                            <label for="username">Username <span class="required">*</span></label>
                            <input type="text" id="username" name="username" class="form-control" required maxlength="20">
                            <span class="field-error" id="username_error"></span>
                            <span class="field-success" id="username_success"></span>
                        </div>
                        
                        <div class="form-group">
                            <label for="password">Password <span class="required">*</span></label>
                            <input type="password" id="password" name="password" class="form-control" required>
                            <div class="password-strength">
                                <div class="strength-bar">
                                    <div class="strength-fill" id="password-strength-fill"></div>
                                </div>
                            </div>
                            <div class="password-requirements">
                                <h6>Password must contain:</h6>
                                <p id="req-lowercase">A lowercase letter</p>
                                <p id="req-uppercase">An uppercase letter</p>
                                <p id="req-number">A number</p>
                                <p id="req-length">At least 6 characters</p>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="repeat_password">Confirm Password <span class="required">*</span></label>
                            <input type="password" id="repeat_password" name="repeat_password" class="form-control" required>
                            <span class="field-error" id="repeat_password_error"></span>
                            <span class="field-success" id="repeat_password_success"></span>
                        </div>
                        
                        <div class="form-group">
                            <div class="checkbox-item">
                                <input type="checkbox" id="terms_agreement" name="terms_agreement" required>
                                <label for="terms_agreement">
                                    I agree to the <a href="terms_condition.php" target="_blank">Terms and Conditions</a>, 
                                    <a href="nda.php" target="_blank">Non-Disclosure Agreement</a>, and 
                                    <a href="medical-disclaimer.php" target="_blank">Medical Disclaimer</a>
                                </label>
                            </div>
                        </div>
                        
                        <div class="btn-container">
                            <button type="button" class="btn btn-secondary" id="prev-step-4">Previous</button>
                            <button type="button" class="btn btn-primary" id="verify-contact">Verify Contact</button>
                        </div>
                    </form>
                </div>
                
                <!-- OTP Verification Section -->
                <div class="section" id="otp-section">
                    <div class="otp-container">
                        <h2>Verify Your Contact</h2>
                        <p>We've sent a verification code to your <span id="otp-method"></span></p>
                        <p><strong id="otp-contact"></strong></p>
                        
                        <div class="otp-inputs">
                            <input type="text" maxlength="1" class="otp-input" data-index="0">
                            <input type="text" maxlength="1" class="otp-input" data-index="1">
                            <input type="text" maxlength="1" class="otp-input" data-index="2">
                            <input type="text" maxlength="1" class="otp-input" data-index="3">
                            <input type="text" maxlength="1" class="otp-input" data-index="4">
                            <input type="text" maxlength="1" class="otp-input" data-index="5">
                        </div>
                        
                        <div id="otp-message" class="alert" style="display: none;"></div>
                        
                        <div class="btn-container">
                            <button type="button" class="btn btn-secondary" id="back-to-account">Back</button>
                            <button type="button" class="btn btn-primary" id="verify-otp">Verify Code</button>
                        </div>
                        
                        <div class="text-center">
                            <span class="otp-resend" id="resend-otp">Didn't receive the code? Try email instead</span>
                            <br><br>
                            <span class="otp-switch" id="switch-otp-method" style="text-decoration: underline; cursor: pointer; color: #007bff;">Try different method (Email/SMS)</span>
                        </div>
                    </div>
                </div>
                
                <!-- Step 5: Address & Subscription -->
                <div class="section" id="step-5">
                    <h2>Billing Address & Subscription</h2>
                    <form id="subscription-form">
                        <!-- Address Section -->
                        <div class="form-group">
                            <label for="billing_address">Address <span class="required">*</span></label>
                            <input type="text" id="billing_address" name="billing_address" class="form-control" required>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="billing_city">City <span class="required">*</span></label>
                                <input type="text" id="billing_city" name="billing_city" class="form-control" required>
                            </div>
                            <div class="form-group">
                                <label for="billing_state">State <span class="required">*</span></label>
                                <input type="text" id="billing_state" name="billing_state" class="form-control" required>
                            </div>
                            <div class="form-group">
                                <label for="billing_zip">Zip Code <span class="required">*</span></label>
                                <input type="text" id="billing_zip" name="billing_zip" class="form-control" required>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="billing_country">Country</label>
                            <select id="billing_country" name="billing_country" class="form-control" required>
                                <option value="USA" selected>United States</option>
                                <option value="Canada">Canada</option>
                                <option value="United Kingdom">United Kingdom</option>
                                <option value="Australia">Australia</option>
                                <option value="France">France</option>
                                <option value="Germany">Germany</option>
                                <option value="Italy">Italy</option>
                                <option value="Spain">Spain</option>
                                <option value="Japan">Japan</option>
                                <option value="South Korea">South Korea</option>
                                <option value="China">China</option>
                                <option value="India">India</option>
                                <option value="Philippines">Philippines</option>
                                <option value="Singapore">Singapore</option>
                                <option value="Malaysia">Malaysia</option>
                                <option value="Thailand">Thailand</option>
                                <option value="Vietnam">Vietnam</option>
                                <option value="Indonesia">Indonesia</option>
                                <option value="Mexico">Mexico</option>
                                <option value="Brazil">Brazil</option>
                            </select>
                        </div>
                        
                        <!-- Subscription Plan -->
                        <div class="form-group">
                            <label>Choose Your Plan</label>
                            <div class="subscription-plans">
                                <div class="plan-card selected" data-plan="plan_1">
                                    <div class="plan-price">$1 <span>/month</span></div>
                                    <div class="plan-title">Doctor AI</div>
                                    <div class="plan-description">Your virtual medical assistant</div>
                                    <input type="radio" name="pricing" value="plan_1" checked style="display: none;">
                                </div>
                            </div>
                        </div>
                        
                        <!-- Coupon Code -->
                        <div class="form-group">
                            <label for="coupon">Apply Coupon <span class="coupon_status"></span></label>
                            <input type="text" id="coupon" name="coupon" class="form-control coupon textbox" placeholder="Enter a valid coupon">
                        </div>
                        
                        <!-- Payment Information -->
                        <div class="payment-section">
                            <div class="submit_payment_credit_card">
                                <div class="form-group">
                                    <label for="card_name">Name on Card</label>
                                    <input type="text" id="card_name" name="card_name" class="form-control">
                                </div>
                        
                        <div class="form-group">
                            <label for="card_email">Email for Receipt</label>
                            <input type="email" id="card_email" name="card_email" class="form-control">
                        </div>
                        
                        <!-- Stripe Elements will be inserted here -->
                        <div class="form-group">
                            <label>Credit or Debit Card</label>
                            <div id="card-element" style="padding: 15px; border: 2px solid #e8eaed; border-radius: 10px; background: white;"></div>
                            <div id="card-errors" role="alert" style="color: #e74c3c; margin-top: 10px;"></div>
                        </div>
                        </div>
                        
                        <div class="submit_payment_coupon" style="display: none;">
                            <div class="form-group">
                                <p style="text-align: center; color: #2ecc71; font-weight: bold;">Valid coupon applied! You can proceed without payment.</p>
                            </div>
                        </div>
                        </div>
                        
                        <div class="btn-container">
                            <div class="submit_payment_credit_card">
                                <button type="button" class="btn btn-secondary" id="prev-step-5">Previous</button>
                                <button type="button" class="btn btn-primary" id="create-account">Create Account</button>
                            </div>
                            <div class="submit_payment_coupon" style="display: none;">
                                <button type="button" class="btn btn-secondary" id="prev-step-5-coupon">Previous</button>
                                <button type="button" class="btn btn-primary verify_coupon">Submit with Coupon</button>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Success Section -->
                <div class="section" id="success-section">
                    <div class="success-animation">
                        <div class="success-icon"></div>
                        <h2>Account Created Successfully!</h2>
                        <p>Welcome to DDxRx! You can now login and start using Doctor AI.</p>
                        <div class="btn-container">
                            <a href="login.php" class="btn btn-primary">Login to Your Account</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="text-center mt-20">
            <p><a href="login.php" class="back-to-login">Cancel and go back to login</a></p>
        </div>
    </div>

    <!-- Loading Overlay -->
    <div id="loading-overlay" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 9999;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: white; padding: 30px; border-radius: 10px; text-align: center;">
            <div class="loading"></div>
            <p style="margin-top: 20px;">Processing...</p>
        </div>
    </div>

    <script>
    $(document).ready(function() {
        let currentStep = 1;
        let maxSteps = 5;
        let otpVerified = false;
        let otpMethod = 'sms';
        let otpSentTime = 0;
        let resendCountdown = 0;
        let countdownInterval = null;
        
        // Initialize Stripe
        const stripe = Stripe(publishable_key);
        const elements = stripe.elements();
        const cardElement = elements.create('card', {
            style: {
                base: {
                    fontSize: '16px',
                    color: '#424770',
                    '::placeholder': {
                        color: '#aab7c4',
                    },
                },
            },
        });
        cardElement.mount('#card-element');
        
        // Card element event handling
        cardElement.on('change', function(event) {
            const displayError = document.getElementById('card-errors');
            if (event.error) {
                displayError.textContent = event.error.message;
            } else {
                displayError.textContent = '';
            }
        });
        
        // Update progress indicator
        function updateProgress() {
            const progressLine = $('.progress-line');
            const steps = $('.step');
            
            steps.removeClass('active completed');
            steps.each(function(index) {
                const stepNum = index + 1;
                if (stepNum < currentStep) {
                    $(this).addClass('completed');
                } else if (stepNum === currentStep) {
                    $(this).addClass('active');
                }
            });
            
            const progress = ((currentStep - 1) / (maxSteps - 1)) * 100;
            progressLine.css('width', progress + '%');
        }
        
        // Show section
        function showSection(step) {
            $('.section').removeClass('active');
            $('#step-' + step + ', #otp-section, #success-section').removeClass('active');
            
            if (step === 'otp') {
                $('#otp-section').addClass('active');
            } else if (step === 'success') {
                $('#success-section').addClass('active');
            } else {
                $('#step-' + step).addClass('active');
            }
        }
        
        // Validation functions
        function validatePersonalInfo() {
            let valid = true;
            const firstName = $('#first_name').val().trim();
            const lastName = $('#last_name').val().trim();
            
            $('.field-error').text('');
            $('.form-control').removeClass('error success');
            
            if (!firstName) {
                $('#first_name_error').text('First name is required');
                $('#first_name').addClass('error');
                valid = false;
            } else {
                $('#first_name').addClass('success');
            }
            
            if (!lastName) {
                $('#last_name_error').text('Last name is required');
                $('#last_name').addClass('error');
                valid = false;
            } else {
                $('#last_name').addClass('success');
            }
            
            return valid;
        }
        
        function validateBirthGender() {
            let valid = true;
            const month = $('#birth_month').val();
            const day = $('#birth_day').val();
            const year = $('#birth_year').val();
            
            $('.field-error').text('');
            
            if (!month || !day || !year) {
                $('#birthday_error').text('Please select your complete date of birth');
                valid = false;
            }
            
            return valid;
        }
        
        function validateContact() {
            let valid = true;
            const phone = $('#phone').val().trim();
            const email = $('#email').val().trim();
            
            // Don't clear existing validation states - preserve them
            
            if (!phone) {
                $('#phone_error').text('Phone number is required');
                $('#phone').removeClass('success').addClass('error');
                valid = false;
            } else {
                // Check if phone has an error from previous validation
                if ($('#phone_error').text().includes('already registered')) {
                    valid = false;
                }
            }
            
            if (!email) {
                $('#email_error').text('Email is required');
                $('#email').removeClass('success').addClass('error');
                valid = false;
            } else if (!isValidEmail(email)) {
                $('#email_error').text('Please enter a valid email address');
                $('#email').removeClass('success').addClass('error');
                valid = false;
            } else {
                // Check if email has an error from previous validation
                if ($('#email_error').text().includes('already registered')) {
                    valid = false;
                }
            }
            
            return valid;
        }
        
        function validateAccount() {
            let valid = true;
            const username = $('#username').val().trim();
            const password = $('#password').val();
            const repeatPassword = $('#repeat_password').val();
            const termsAgreed = $('#terms_agreement').is(':checked');
            
            $('.field-error').text('');
            $('.form-control').removeClass('error success');
            
            if (!username) {
                $('#username_error').text('Username is required');
                $('#username').addClass('error');
                valid = false;
            } else if (username.length < 3) {
                $('#username_error').text('Username must be at least 3 characters');
                $('#username').addClass('error');
                valid = false;
            }
            
            if (!password) {
                $('#password').addClass('error');
                valid = false;
            } else if (!isPasswordValid(password)) {
                $('#password').addClass('error');
                valid = false;
            }
            
            if (!repeatPassword) {
                $('#repeat_password_error').text('Please confirm your password');
                $('#repeat_password').addClass('error');
                valid = false;
            } else if (password !== repeatPassword) {
                $('#repeat_password_error').text('Passwords do not match');
                $('#repeat_password').addClass('error');
                valid = false;
            } else if (password === repeatPassword && password) {
                $('#repeat_password_success').text('Passwords match');
                $('#repeat_password').addClass('success');
            }
            
            if (!termsAgreed) {
                showAlert('error', 'Please agree to the terms and conditions');
                valid = false;
            }
            
            return valid;
        }
        
        function isValidEmail(email) {
            const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        }
        
        function isPasswordValid(password) {
            const hasLower = /[a-z]/.test(password);
            const hasUpper = /[A-Z]/.test(password);
            const hasNumber = /\d/.test(password);
            const hasLength = password.length >= 6;
            
            return hasLower && hasUpper && hasNumber && hasLength;
        }
        
        function updatePasswordStrength(password) {
            const hasLower = /[a-z]/.test(password);
            const hasUpper = /[A-Z]/.test(password);
            const hasNumber = /\d/.test(password);
            const hasLength = password.length >= 6;
            
            $('#req-lowercase').toggleClass('valid', hasLower);
            $('#req-uppercase').toggleClass('valid', hasUpper);
            $('#req-number').toggleClass('valid', hasNumber);
            $('#req-length').toggleClass('valid', hasLength);
            
            const score = [hasLower, hasUpper, hasNumber, hasLength].filter(Boolean).length;
            const strengthFill = $('#password-strength-fill');
            
            strengthFill.removeClass('weak medium strong');
            if (score <= 1) {
                strengthFill.addClass('weak');
            } else if (score <= 2) {
                strengthFill.addClass('medium');
            } else if (score >= 3) {
                strengthFill.addClass('strong');
            }
        }
        
        function showAlert(type, message) {
            if (type === 'error') {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: message,
                    confirmButtonColor: '#007bff'
                });
            } else if (type === 'success') {
                Swal.fire({
                    icon: 'success',
                    title: 'Success',
                    text: message,
                    confirmButtonColor: '#007bff'
                });
            } else if (type === 'info') {
                Swal.fire({
                    icon: 'info',
                    title: 'Information',
                    text: message,
                    confirmButtonColor: '#007bff'
                });
            } else {
                Swal.fire({
                    title: 'Notice',
                    text: message,
                    confirmButtonColor: '#007bff'
                });
            }
        }
        
        function showLoading() {
            $('#loading-overlay').show();
        }
        
        function hideLoading() {
            $('#loading-overlay').hide();
        }
        
        // Event handlers
        $('#next-step-1').click(function() {
            if (validatePersonalInfo()) {
                currentStep = 2;
                showSection(2);
                updateProgress();
            }
        });
        
        $('#prev-step-2, #next-step-2').click(function() {
            if ($(this).attr('id') === 'prev-step-2') {
                currentStep = 1;
                showSection(1);
            } else if (validateBirthGender()) {
                currentStep = 3;
                showSection(3);
            }
            updateProgress();
        });
        
        $('#prev-step-3, #next-step-3').click(function() {
            if ($(this).attr('id') === 'prev-step-3') {
                currentStep = 2;
                showSection(2);
            } else if (validateContact()) {
                currentStep = 4;
                showSection(4);
            }
            updateProgress();
        });
        
        $('#prev-step-4').click(function() {
            currentStep = 3;
            showSection(3);
            updateProgress();
        });
        
        $('#verify-contact').click(function() {
            if (validateAccount()) {
                // Check username and email availability first
                checkUsernameAndEmail();
            }
        });
        
        $('#back-to-account').click(function() {
            showSection(4);
        });
        
        $('#verify-otp').click(function() {
            verifyOTP();
        });
        
        $('#resend-otp').click(function() {
            // Don't allow click if countdown is active
            if (resendCountdown > 0) {
                return;
            }
            
            // Clear any existing countdown
            clearResendCountdown();
            
            // Resend OTP with the same method first, then offer alternative
            sendOTP(otpMethod);
        });
        
        $('#switch-otp-method').click(function() {
            // Clear any existing countdown
            clearResendCountdown();
            
            // Toggle between SMS and email
            otpMethod = otpMethod === 'sms' ? 'email' : 'sms';
            sendOTP(otpMethod);
        });
        
        $('#prev-step-5').click(function() {
            currentStep = 4;
            showSection(4);
            updateProgress();
        });
        
        $('#create-account').click(function() {
            if (otpVerified) {
                createAccount();
            } else {
                showAlert('error', 'Please verify your contact information first');
            }
        });
        
        // Real-time validation
        $('#password').on('input', function() {
            updatePasswordStrength($(this).val());
        });
        
        $('#username').on('blur', function() {
            const username = $(this).val().trim();
            if (username.length >= 3) {
                checkUsername(username);
            }
        });
        
        $('#email').on('blur', function() {
            const email = $(this).val().trim();
            if (isValidEmail(email)) {
                checkEmail(email);
            }
        });
        
        $('#phone').on('blur', function() {
            const phone = $(this).val().trim();
            const country_code = $('#country_code').val();
            if (phone.length >= 7) { // Minimum phone number length
                checkPhone(phone, country_code);
            }
        });
        
        $('#country_code').on('change', function() {
            const phone = $('#phone').val().trim();
            const country_code = $(this).val();
            if (phone.length >= 7) {
                checkPhone(phone, country_code);
            }
        });
        
        // Radio button handlers
        $('.radio-item').click(function() {
            const group = $(this).closest('.radio-group');
            group.find('.radio-item').removeClass('selected');
            $(this).addClass('selected');
            $(this).find('input[type="radio"]').prop('checked', true);
        });
        
        // OTP input handling
        $('.otp-input').on('input', function() {
            const value = $(this).val();
            if (value.length === 1) {
                const nextInput = $(this).closest('.otp-inputs').find('.otp-input').eq($(this).data('index') + 1);
                if (nextInput.length) {
                    nextInput.focus();
                }
            }
        });
        
        $('.otp-input').on('keydown', function(e) {
            if (e.key === 'Backspace' && $(this).val() === '') {
                const prevInput = $(this).closest('.otp-inputs').find('.otp-input').eq($(this).data('index') - 1);
                if (prevInput.length) {
                    prevInput.focus();
                }
            }
        });
        
        // Enter key handlers for form progression
        $('#step-1 input').on('keypress', function(e) {
            if (e.which === 13) { // Enter key
                e.preventDefault();
                $('#next-step-1').click();
            }
        });
        
        $('#step-2 select').on('keypress', function(e) {
            if (e.which === 13) { // Enter key
                e.preventDefault();
                $('#next-step-2').click();
            }
        });
        
        $('#step-3 input, #step-3 select').on('keypress', function(e) {
            if (e.which === 13) { // Enter key
                e.preventDefault();
                $('#next-step-3').click();
            }
        });
        
        $('#step-4 input').on('keypress', function(e) {
            if (e.which === 13) { // Enter key
                e.preventDefault();
                if (otpVerified) {
                    showSection(5);
                    currentStep = 5;
                    updateProgress();
                } else {
                    showAlert('error', 'Please verify your contact information first');
                }
            }
        });
        
        $('.otp-input').on('keypress', function(e) {
            if (e.which === 13) { // Enter key
                e.preventDefault();
                $('#verify-otp').click();
            }
        });
        
        function checkUsernameAndEmail() {
            showLoading();
            const username = $('#username').val().trim();
            const email = $('#email').val().trim();
            
            // Check username first
            $.post('', {
                action: 'check_username',
                username: username,
                csrf_token: csrf_token
            }, function(response) {
                const data = JSON.parse(response);
                if (!data.available) {
                    hideLoading();
                    $('#username_error').text(data.message);
                    $('#username').addClass('error');
                    return;
                }
                
                // Check email
                $.post('', {
                    action: 'check_email',
                    email: email,
                    csrf_token: csrf_token
                }, function(response) {
                    const data = JSON.parse(response);
                    if (!data.available) {
                        hideLoading();
                        $('#email_error').text(data.message);
                        $('#email').addClass('error');
                        return;
                    }
                    
                    // Both available, proceed to OTP
                    hideLoading();
                    sendOTP('sms');
                });
            });
        }
        
        function checkUsername(username) {
            $.post('', {
                action: 'check_username',
                username: username,
                csrf_token: csrf_token
            }, function(response) {
                const data = JSON.parse(response);
                $('#username_error, #username_success').text('');
                $('#username').removeClass('error success');
                
                if (data.available) {
                    $('#username_success').text(data.message);
                    $('#username').addClass('success');
                } else {
                    $('#username_error').text(data.message);
                    $('#username').addClass('error');
                }
            });
        }
        
        function checkEmail(email) {
            $.post('', {
                action: 'check_email',
                email: email,
                csrf_token: csrf_token
            }, function(response) {
                const data = JSON.parse(response);
                $('#email_error, #email_success').text('');
                $('#email').removeClass('error success');
                
                if (data.available) {
                    $('#email_success').text(data.message);
                    $('#email').addClass('success');
                } else {
                    $('#email_error').text(data.message);
                    $('#email').addClass('error');
                }
            });
        }
        
        function checkPhone(phone, country_code) {
            $.post('', {
                action: 'check_phone',
                phone: phone,
                country_code: country_code,
                csrf_token: csrf_token
            }, function(response) {
                const data = JSON.parse(response);
                $('#phone_error, #phone_success').text('');
                $('#phone').removeClass('error success');
                
                if (data.available) {
                    $('#phone_success').text(data.message);
                    $('#phone').addClass('success');
                } else {
                    $('#phone_error').text(data.message);
                    $('#phone').addClass('error');
                }
            });
        }
        
        function sendOTP(method) {
            showLoading();
            const phone = $('#phone').val().trim();
            const country_code = $('#country_code').val();
            const email = $('#email').val().trim();
            
            $.post('', {
                action: 'send_otp',
                phone: phone,
                country_code: country_code,
                email: email,
                method: method,
                csrf_token: csrf_token
            }, function(response) {
                hideLoading();
                const data = JSON.parse(response);
                
                if (data.success) {
                    showSection('otp');
                    $('#otp-method').text(method === 'sms' ? 'phone' : 'email');
                    $('#otp-contact').text(method === 'sms' ? country_code + phone : email);
                    $('#resend-otp').text(method === 'sms' ? 
                        "Didn't receive the code? Resend SMS" : 
                        "Didn't receive the code? Resend Email"
                    );
                    $('#switch-otp-method').text(method === 'sms' ? 
                        "Try Email instead" : 
                        "Try SMS instead"
                    );
                    otpMethod = method;
                    
                    // Start countdown timer for SMS resend
                    if (method === 'sms') {
                        startResendCountdown();
                    }
                } else {
                    showAlert('error', data.message);
                }
            });
        }
        
        function verifyOTP() {
            const otpInputs = $('.otp-input');
            let otp = '';
            otpInputs.each(function() {
                otp += $(this).val();
            });
            
            if (otp.length !== 6) {
                showOTPMessage('error', 'Please enter the complete 6-digit code');
                return;
            }
            
            showLoading();
            $.post('', {
                action: 'verify_otp',
                otp: otp,
                csrf_token: csrf_token
            }, function(response) {
                hideLoading();
                const data = JSON.parse(response);
                
                if (data.success) {
                    otpVerified = true;

                    
                    /*
                    currentStep = 5;
                    showSection(5);
                    updateProgress();*/
                    
                    // Pre-fill email for receipt
                    $('#card_email').val($('#email').val());
                    submitAccountData();
                } else {
                    showOTPMessage('error', data.message);
                    // Clear OTP inputs
                    $('.otp-input').val('');
                    $('.otp-input').first().focus();
                }
            });
        }
        
        function showOTPMessage(type, message) {
            const messageDiv = $('#otp-message');
            messageDiv.removeClass('alert-success alert-error alert-info')
                     .addClass('alert-' + type)
                     .text(message)
                     .show();
            
            setTimeout(function() {
                messageDiv.hide();
            }, 5000);
        }
        
        function startResendCountdown() {
            resendCountdown = 120; // 2 minutes in seconds
            otpSentTime = Date.now();
            
            const resendButton = $('#resend-otp');
            resendButton.addClass('disabled').css('pointer-events', 'none');
            
            countdownInterval = setInterval(function() {
                resendCountdown--;
                
                const minutes = Math.floor(resendCountdown / 60);
                const seconds = resendCountdown % 60;
                const timeString = minutes + ':' + (seconds < 10 ? '0' : '') + seconds;
                
                if (otpMethod === 'sms') {
                    resendButton.text('Resend SMS in ' + timeString);
                } else {
                    resendButton.text('Resend Email in ' + timeString);
                }
                
                if (resendCountdown <= 0) {
                    clearInterval(countdownInterval);
                    resendButton.removeClass('disabled').css('pointer-events', 'auto');
                    
                    if (otpMethod === 'sms') {
                        resendButton.text("Didn't receive the code? Resend SMS");
                    } else {
                        resendButton.text("Didn't receive the code? Resend Email");
                    }
                }
            }, 1000);
        }
        
        function clearResendCountdown() {
            if (countdownInterval) {
                clearInterval(countdownInterval);
                countdownInterval = null;
            }
            resendCountdown = 0;
        }
        
        function createAccount() {
            // Check if using coupon or credit card
            const isCouponActive = $('.submit_payment_coupon').is(':visible');
            
            if (!isCouponActive) {
                // Validate credit card fields
                const cardName = $('#card_name').val().trim();
                const cardEmail = $('#card_email').val().trim();
                
                if (!cardName) {
                    showAlert('error', 'Name on card is required');
                    return;
                }
                
                if (!cardEmail) {
                    showAlert('error', 'Email for receipt is required');
                    return;
                }
                
                if (!isValidEmail(cardEmail)) {
                    showAlert('error', 'Please enter a valid email for receipt');
                    return;
                }
                
                // Check if Stripe card element is complete
                if (!stripe || !cardElement) {
                    showAlert('error', 'Please enter valid credit card information');
                    return;
                }
            }
            if (!isCouponActive) {
                // Process credit card payment
                processStripePayment();
            } else {
                // Process account creation without payment (coupon)
                submitAccountData();
            }
        }
        
        function processStripePayment() {
            // Validate billing address fields
            const billingAddress = $('#billing_address').val().trim();
            const billingCity = $('#billing_city').val().trim();
            const billingState = $('#billing_state').val().trim();
            const billingZip = $('#billing_zip').val().trim();
            
            if (!billingAddress) {
                showAlert('error', 'Billing address is required');
                return;
            }
            
            if (!billingCity) {
                showAlert('error', 'Billing city is required');
                return;
            }
            
            if (!billingState) {
                showAlert('error', 'Billing state is required');
                return;
            }
            
            if (!billingZip) {
                showAlert('error', 'Billing zip code is required');
                return;
            }
            
            showLoading();
            
            const cardName = $('#card_name').val().trim();
            const cardEmail = $('#card_email').val().trim();
            
            // Create Stripe token
            stripe.createToken(cardElement, {
                name: cardName,
                address_line1: billingAddress,
                address_city: billingCity,
                address_state: billingState,
                address_zip: billingZip,
                address_country: $('#billing_country').val()
            }).then(function(result) {
                hideLoading();
                
                if (result.error) {
                    showAlert('error', result.error.message);
                } else {
                    // Submit account data with Stripe token
                    submitAccountData(result.token.id, cardEmail);
                }
            });
        }
        
        function submitAccountData(stripeToken = null, stripeEmail = null) {
            showLoading();
            
            // Collect all form data
            const formData = {
                action: 'create_account',
                csrf_token: csrf_token,
                first_name: $('#first_name').val().trim(),
                middle_name: $('#middle_name').val().trim(),
                last_name: $('#last_name').val().trim(),
                birthday: $('#birth_year').val() + '-' + $('#birth_month').val() + '-' + $('#birth_day').val(),
                gender: $('input[name="gender"]:checked').val(),
                location: $('input[name="location"]:checked').val(),
                phone: $('#phone').val().trim(),
                country_code: $('#country_code').val(),
                email: $('#email').val().trim(),
                username: $('#username').val().trim(),
                password: $('#password').val(),
                billing_address: $('#billing_address').val().trim(),
                billing_city: $('#billing_city').val().trim(),
                billing_state: $('#billing_state').val().trim(),
                billing_zip: $('#billing_zip').val().trim(),
                billing_country: $('#billing_country').val(),
                coupon_code: $('#coupon').val()
            };
            
            // Add Stripe data if paying with credit card
            if (stripeToken) {
                formData.stripeToken = stripeToken;
                formData.stripeEmail = stripeEmail;
                formData.pricing = 'plan_1'; // $1/month plan as mentioned in PR description
            }
            
            $.post('', formData, function(response) {
                hideLoading();
                const data = JSON.parse(response);
                
                if (data.success) {
                    showSection('success');
                    currentStep = maxSteps;
                    updateProgress();
                } else {
                    showAlert('error', data.message);
                }
            });
        }
        
        // Coupon functionality
        $('.textbox.coupon').on('input', function(e) {
            var coupon_code = $(this).val();
            
            $.ajax({
                type: "POST",
                url: 'process/check_coupon_api.php',
                data: {
                    coupon_code: coupon_code,
                },
                beforeSend: function() {
                    $('.submit_payment_credit_card').show();
                    $('.submit_payment_coupon').hide();
                    $('.coupon_status').removeClass('valid');
                    $('.coupon_status').removeClass('invalid');
                    $('.coupon_status').html('');
                },
                success: function(data) {
                    if (data.result === true) {
                        $('.coupon_status').removeClass('invalid');
                        $('.coupon_status').addClass('valid');
                        $('.coupon_status').html('Coupon Code Valid');
                        $('.submit_payment_credit_card').hide();
                        $('.submit_payment_coupon').show();
                    } else {
                        $('.coupon_status').removeClass('valid');
                        $('.coupon_status').addClass('invalid');
                        $('.coupon_status').html('Coupon Code Invalid');
                    }
                },
                dataType: 'json',
                fail: function(response) {
                    console.log('Coupon check failed');
                }
            });
        });
        
        // Coupon submission
        $('.verify_coupon').click(function() {
            showLoading();
            
            const formData = {
                action: 'create_account',
                csrf_token: csrf_token,
                first_name: $('#first_name').val().trim(),
                middle_name: $('#middle_name').val().trim(),
                last_name: $('#last_name').val().trim(),
                birthday: $('#birth_year').val() + '-' + $('#birth_month').val() + '-' + $('#birth_day').val(),
                gender: $('input[name="gender"]:checked').val(),
                location: $('input[name="location"]:checked').val(),
                phone: $('#phone').val().trim(),
                country_code: $('#country_code').val(),
                email: $('#email').val().trim(),
                username: $('#username').val().trim(),
                password: $('#password').val(),
                billing_address: $('#billing_address').val().trim(),
                billing_city: $('#billing_city').val().trim(),
                billing_state: $('#billing_state').val().trim(),
                billing_zip: $('#billing_zip').val().trim(),
                billing_country: $('#billing_country').val(),
                coupon_code: $('#coupon').val()
            };
            
            $.post('', formData, function(response) {
                hideLoading();
                const data = JSON.parse(response);
                
                if (data.success) {
                    showSection('success');
                    setTimeout(function() {
                        window.location.href = 'patient_login.php';
                    }, 3000);
                } else {
                    showAlert('error', data.message);
                }
            }).fail(function() {
                hideLoading();
                showAlert('error', 'An error occurred. Please try again.');
            });
        });
        
        // Previous button for coupon section
        $('#prev-step-5-coupon').click(function() {
            currentStep = 4;
            showSection(currentStep);
            updateProgress();
        });
        
        // Real-time password validation
        $('#password').on('input keyup', function() {
            checkPasswordStrength($(this).val());
            checkPasswordMatch();
        });
        
        $('#repeat_password').on('input keyup', function() {
            checkPasswordMatch();
        });
        
        function checkPasswordMatch() {
            const password = $('#password').val();
            const repeatPassword = $('#repeat_password').val();
            const errorElement = $('#repeat_password_error');
            const successElement = $('#repeat_password_success');
            const inputElement = $('#repeat_password');
            
            // Clear previous states
            errorElement.text('');
            successElement.text('');
            inputElement.removeClass('error success');
            
            if (repeatPassword === '') {
                return; // Don't show error for empty field
            }
            
            if (password !== repeatPassword) {
                errorElement.text('Passwords do not match');
                inputElement.addClass('error');
            } else if (password === repeatPassword && password !== '') {
                successElement.text('Passwords match');
                inputElement.addClass('success');
            }
        }
        
        // Initialize
        updateProgress();
    });
    </script>
</body>
</html>
