<?php
/**
 * Stripe PHP SDK initialization and fallback stubs
 * Compatible with both PHP 5.6 and PHP 8.2
 */

// First try the vendor autoloader (preferred method)
if (file_exists(__DIR__ . '/../../vendor/autoload.php')) {
    require_once __DIR__ . '/../../vendor/autoload.php';
    // Check if Stripe is available after autoloader
    if (class_exists('Stripe\Stripe')) {
        return;
    }
}

// Try the stripe-php-7.80.0 directory
if (file_exists(__DIR__ . '/../stripe-php-7.80.0/init.php')) {
    require_once __DIR__ . '/../stripe-php-7.80.0/init.php';
    return;
}

// If no proper Stripe installation is found, we need to handle this gracefully
// For live mode, this should fail; for test mode, we can provide stubs

// Create the basic Stripe namespace structure with better error handling
if (!class_exists('Stripe\Stripe')) {
    
    // Simple fallback classes that handle both test and live modes appropriately
    class Stripe_Stripe {
        private static $apiKey = null;
        
        public static function setApiKey($key) {
            self::$apiKey = $key;
        }
        
        public static function getApiKey() {
            return self::$apiKey;
        }
        
        public static function isTestMode() {
            return self::$apiKey && (
                strpos(self::$apiKey, 'sk_test_') === 0 || 
                strpos(self::$apiKey, 'test') !== false
            );
        }
    }
    
    class Stripe_Customer {
        public static function create($params) {
            // Only allow test mode operations
            if (!Stripe_Stripe::isTestMode()) {
                throw new Exception('Stripe SDK not properly installed. Please install the Stripe PHP SDK for live payments.');
            }
            
            // Return mock response for test mode
            return (object) array(
                'id' => 'cus_test_' . uniqid(),
                'object' => 'customer',
                'email' => isset($params['email']) ? $params['email'] : '',
                'created' => time(),
                'subscriptions' => (object) array('data' => array())
            );
        }
    }
    
    class Stripe_Subscription {
        public static function create($params) {
            // Only allow test mode operations
            if (!Stripe_Stripe::isTestMode()) {
                throw new Exception('Stripe SDK not properly installed. Please install the Stripe PHP SDK for live payments.');
            }
            
            // Return mock response for test mode
            return (object) array(
                'id' => 'sub_test_' . uniqid(),
                'object' => 'subscription',
                'customer' => isset($params['customer']) ? $params['customer'] : '',
                'status' => 'active',
                'current_period_end' => time() + (30 * 24 * 60 * 60), // 30 days from now
                'created' => time()
            );
        }
    }
    
    // Exception classes
    class Stripe_Exception extends Exception {}
    class Stripe_Error_Card extends Stripe_Exception {}
    class Stripe_Error_InvalidRequest extends Stripe_Exception {}
    class Stripe_Error_Authentication extends Stripe_Exception {}
    class Stripe_Error_ApiConnection extends Stripe_Exception {}
    class Stripe_Error_Api extends Stripe_Exception {}
    class Stripe_Error_RateLimit extends Stripe_Exception {}
    
    // Create namespace aliases for modern PHP code
    if (!class_exists('Stripe\Stripe')) {
        class_alias('Stripe_Stripe', 'Stripe\Stripe');
        class_alias('Stripe_Customer', 'Stripe\Customer');
        class_alias('Stripe_Subscription', 'Stripe\Subscription');
        
        // Exception aliases - avoid duplicate ApiErrorException
        if (!class_exists('Stripe\Exception\ApiErrorException')) {
            class_alias('Stripe_Exception', 'Stripe\Exception\ApiErrorException');
        }
        if (!class_exists('Stripe\Exception\CardException')) {
            class_alias('Stripe_Error_Card', 'Stripe\Exception\CardException');
        }
        if (!class_exists('Stripe\Exception\InvalidRequestException')) {
            class_alias('Stripe_Error_InvalidRequest', 'Stripe\Exception\InvalidRequestException');
        }
        if (!class_exists('Stripe\Exception\AuthenticationException')) {
            class_alias('Stripe_Error_Authentication', 'Stripe\Exception\AuthenticationException');
        }
        if (!class_exists('Stripe\Exception\ApiConnectionException')) {
            class_alias('Stripe_Error_ApiConnection', 'Stripe\Exception\ApiConnectionException');
        }
        if (!class_exists('Stripe\Exception\RateLimitException')) {
            class_alias('Stripe_Error_RateLimit', 'Stripe\Exception\RateLimitException');
        }
    }
}