<?php
/**
 * PointClickCare Integration Helper Functions
 * 
 * This file contains all helper functions needed for PCC OAuth integration,
 * user management, and data synchronization.
 * 
 * PHP 5.6/8.1 Compatible
 */

// Load PCC configuration
require_once(dirname(__FILE__) . '/../pcc-config.php');

// Load DDxRx functions
require_once(dirname(__FILE__) . '/../../process/functions.php');

/**
 * Exchange authorization code for access token
 * 
 * @param string $code Authorization code from PCC
 * @return array|false Array with token data or false on failure
 */
function pcc_exchange_code_for_token($code) {
    pcc_log('Starting token exchange for authorization code', 'DEBUG');
    
    // Prepare POST data
    $post_data = array(
        'grant_type' => 'authorization_code',
        'code' => $code,
        'redirect_uri' => PCC_OAUTH_CALLBACK_URL,
        'client_id' => PCC_CUSTOMER_KEY,
        'client_secret' => PCC_CUSTOMER_SECRET
    );
    
    // Initialize cURL
    $ch = curl_init(PCC_TOKEN_URL);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post_data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: application/x-www-form-urlencoded',
        'Accept: application/json'
    ));
    
    // Execute request
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    // Check for cURL errors
    if ($curl_error) {
        pcc_log('cURL error during token exchange: ' . $curl_error, 'ERROR');
        return false;
    }
    
    // Check HTTP response code
    if ($http_code !== 200) {
        pcc_log('Token exchange failed with HTTP code: ' . $http_code . ' Response: ' . $response, 'ERROR');
        return false;
    }
    
    // Parse JSON response
    $token_data = json_decode($response, true);
    
    if (!$token_data || !isset($token_data['access_token'])) {
        pcc_log('Invalid token response: ' . $response, 'ERROR');
        return false;
    }
    
    pcc_log('Token exchange successful', 'INFO');
    
    return $token_data;
}

/**
 * Fetch user data from PCC API using access token
 * Note: PCC API requires orgId and userId in URL, but token response should contain user info
 * 
 * @param string $access_token PCC access token
 * @param array $token_data Full token response data (may contain user info)
 * @return array|false User data array or false on failure
 */
function pcc_get_user_data($access_token, $token_data = null) {
    pcc_log('Fetching user data from PCC API', 'DEBUG');
    
    // First, check if user info is already in the token response
    if ($token_data && isset($token_data['user'])) {
        pcc_log('User data found in token response', 'DEBUG');
        return $token_data['user'];
    }
    
    // If we have orgUuid from token metadata, we can use the userinfo endpoint
    if ($token_data && isset($token_data['metadata']) && isset($token_data['metadata']['orgUuid'])) {
        $user_info_url = str_replace('{orgUuid}', $token_data['metadata']['orgUuid'], PCC_USER_INFO_URL);
        
        pcc_log('Attempting to fetch user data from: ' . $user_info_url, 'DEBUG');
        
        // Initialize cURL
        $ch = curl_init($user_info_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Authorization: Bearer ' . $access_token,
            'Accept: application/json'
        ));
        
        // Execute request
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        // Check for cURL errors
        if ($curl_error) {
            pcc_log('cURL error during user data fetch: ' . $curl_error, 'ERROR');
            return false;
        }
        
        // Check HTTP response code
        if ($http_code !== 200) {
            pcc_log('User data fetch failed with HTTP code: ' . $http_code . ' Response: ' . $response, 'ERROR');
            return false;
        }
        
        // Parse JSON response
        $user_data = json_decode($response, true);
        
        if (!$user_data) {
            pcc_log('Invalid user data response: ' . $response, 'ERROR');
            return false;
        }
        
        pcc_log('User data fetched successfully from API', 'DEBUG');
        return $user_data;
    }
    
    // If we can't get user data from API, try to extract basic info from token
    pcc_log('Cannot fetch user data from API - missing orgUuid in token metadata', 'WARNING');
    
    // Create minimal user data from token if available
    $minimal_user_data = array();
    if ($token_data) {
        if (isset($token_data['access_token'])) $minimal_user_data['access_token'] = $token_data['access_token'];
        if (isset($token_data['metadata']['orgUuid'])) $minimal_user_data['orgUuid'] = $token_data['metadata']['orgUuid'];
        if (isset($token_data['scope'])) $minimal_user_data['scope'] = $token_data['scope'];
        if (isset($token_data['expires_in'])) $minimal_user_data['expires_in'] = $token_data['expires_in'];
    }
    
    if (!empty($minimal_user_data)) {
        pcc_log('Using minimal user data from token: ' . json_encode($minimal_user_data), 'INFO');
        return $minimal_user_data;
    }
    
    pcc_log('No user data available in token response', 'ERROR');
    return false;
}

/**
 * Map PCC role to DDxRx user type
 * 
 * @param string $pcc_role PCC role identifier
 * @return string DDxRx user type (0-4)
 */
function pcc_map_role_to_usertype($pcc_role) {
    $role_mapping = pcc_get_role_mapping();
    
    // Convert role to lowercase for case-insensitive matching
    $pcc_role_lower = strtolower(trim($pcc_role));
    
    // Check if role exists in mapping
    if (isset($role_mapping[$pcc_role_lower])) {
        $user_type = $role_mapping[$pcc_role_lower];
        pcc_log('Mapped PCC role "' . $pcc_role . '" to DDxRx userType: ' . $user_type, 'DEBUG');
        return $user_type;
    }
    
    // Return default user type (Patient)
    $default_type = $role_mapping['default'];
    pcc_log('Unknown PCC role "' . $pcc_role . '", using default userType: ' . $default_type, 'WARNING');
    
    return $default_type;
}

/**
 * Check if PCC user exists in DDxRx database
 * 
 * @param string $pcc_uuid PCC user UUID
 * @return array|false User data if exists, false otherwise
 */
function pcc_check_user_exists($pcc_uuid) {
    $con = pcc_db_connect();
    
    if (!$con) {
        return false;
    }
    
    // Prepare statement to prevent SQL injection
    $stmt = mysqli_prepare($con, "SELECT * FROM all_users WHERE pcc_user_uuid = ?");
    
    if (!$stmt) {
        pcc_log('Failed to prepare statement: ' . mysqli_error($con), 'ERROR');
        mysqli_close($con);
        return false;
    }
    
    mysqli_stmt_bind_param($stmt, "s", $pcc_uuid);
    mysqli_stmt_execute($stmt);
    
    $result = mysqli_stmt_get_result($stmt);
    $user = mysqli_fetch_assoc($result);
    
    mysqli_stmt_close($stmt);
    mysqli_close($con);
    
    if ($user) {
        pcc_log('PCC user found in database: ' . $pcc_uuid, 'DEBUG');
        return $user;
    }
    
    pcc_log('PCC user not found in database: ' . $pcc_uuid, 'DEBUG');
    return false;
}

/**
 * Create new DDxRx user from PCC data
 * 
 * @param array $pcc_user_data User data from PCC API
 * @return int|false User ID if successful, false on failure
 */
function pcc_create_ddxrx_user($pcc_user_data) {
    pcc_log('Creating new DDxRx user from PCC data', 'INFO');
    
    $con = pcc_db_connect();
    
    if (!$con) {
        return false;
    }
    
    // Extract user information from PCC data
    // Note: Adjust these fields based on actual PCC API response structure
    $pcc_uuid = isset($pcc_user_data['id']) ? $pcc_user_data['id'] : '';
    $first_name = isset($pcc_user_data['firstName']) ? $pcc_user_data['firstName'] : '';
    $last_name = isset($pcc_user_data['lastName']) ? $pcc_user_data['lastName'] : '';
    $email = isset($pcc_user_data['email']) ? $pcc_user_data['email'] : '';
    $pcc_role = isset($pcc_user_data['role']) ? $pcc_user_data['role'] : 'testuser';
    $username = isset($pcc_user_data['username']) ? $pcc_user_data['username'] : $email;
    
    // Validate required fields
    if (empty($pcc_uuid) || empty($email) || empty($first_name) || empty($last_name)) {
        pcc_log('Missing required user data from PCC', 'ERROR');
        mysqli_close($con);
        return false;
    }
    
    // Map PCC role to DDxRx user type
    $user_type = pcc_map_role_to_usertype($pcc_role);
    
    // Generate secure password (user won't use it for login)
    $password = pcc_generate_password(PCC_PASSWORD_LENGTH);
    $password_hash = md5($password); // Using MD5 as per existing codebase
    
    // Get configuration values
    $group_id = PCC_DEFAULT_GROUP_ID;
    $location_id = PCC_DEFAULT_LOCATION_ID;
    $created_at = date('Y-m-d H:i:s');
    $is_active = '1';
    
    // Start transaction
    mysqli_begin_transaction($con);
    
    try {
        // Insert into all_users table
        $stmt = mysqli_prepare($con, 
            "INSERT INTO all_users (groupid, username, password, email_address, user_type, location_id, is_active, created_at, pcc_user_uuid) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
        );
        
        if (!$stmt) {
            throw new Exception('Failed to prepare all_users insert: ' . mysqli_error($con));
        }
        
        mysqli_stmt_bind_param($stmt, "sssssssss", 
            $group_id, $username, $password_hash, $email, $user_type, 
            $location_id, $is_active, $created_at, $pcc_uuid
        );
        
        if (!mysqli_stmt_execute($stmt)) {
            throw new Exception('Failed to insert into all_users: ' . mysqli_stmt_error($stmt));
        }
        
        $user_id = mysqli_insert_id($con);
        mysqli_stmt_close($stmt);
        
        pcc_log('Created all_users record with ID: ' . $user_id, 'DEBUG');
        
        // Insert into appropriate type-specific table
        if ($user_type == '2') {
            // Create physician record
            $stmt = mysqli_prepare($con,
                "INSERT INTO physician (user_id, groupid, physician_fname, physician_lname, physician_email, physician_active, created_at) 
                 VALUES (?, ?, ?, ?, ?, '1', ?)"
            );
            
            if (!$stmt) {
                throw new Exception('Failed to prepare physician insert: ' . mysqli_error($con));
            }
            
            mysqli_stmt_bind_param($stmt, "isssss", 
                $user_id, $group_id, $first_name, $last_name, $email, $created_at
            );
            
            if (!mysqli_stmt_execute($stmt)) {
                throw new Exception('Failed to insert into physician: ' . mysqli_stmt_error($stmt));
            }
            
            mysqli_stmt_close($stmt);
            pcc_log('Created physician record for user ID: ' . $user_id, 'DEBUG');
            
        } elseif ($user_type == '4') {
            // Create nurse record
            $stmt = mysqli_prepare($con,
                "INSERT INTO nurse (user_id, groupid, nurse_fname, nurse_lname, nurse_email, nurse_active, created_at) 
                 VALUES (?, ?, ?, ?, ?, '1', ?)"
            );
            
            if (!$stmt) {
                throw new Exception('Failed to prepare nurse insert: ' . mysqli_error($con));
            }
            
            mysqli_stmt_bind_param($stmt, "isssss", 
                $user_id, $group_id, $first_name, $last_name, $email, $created_at
            );
            
            if (!mysqli_stmt_execute($stmt)) {
                throw new Exception('Failed to insert into nurse: ' . mysqli_stmt_error($stmt));
            }
            
            mysqli_stmt_close($stmt);
            pcc_log('Created nurse record for user ID: ' . $user_id, 'DEBUG');
            
        } elseif ($user_type == '3') {
            // Create patient record
            $stmt = mysqli_prepare($con,
                "INSERT INTO patients (user_id, groupid, location_id, patient_fname, patient_lname, email_id, active, created_at) 
                 VALUES (?, ?, ?, ?, ?, ?, '1', ?)"
            );
            
            if (!$stmt) {
                throw new Exception('Failed to prepare patient insert: ' . mysqli_error($con));
            }
            
            mysqli_stmt_bind_param($stmt, "isissss", 
                $user_id, $group_id, $location_id, $first_name, $last_name, $email, $created_at
            );
            
            if (!mysqli_stmt_execute($stmt)) {
                throw new Exception('Failed to insert into patients: ' . mysqli_stmt_error($stmt));
            }
            
            mysqli_stmt_close($stmt);
            pcc_log('Created patient record for user ID: ' . $user_id, 'DEBUG');
        }
        
        // Commit transaction
        mysqli_commit($con);
        
        pcc_log('Successfully created DDxRx user with ID: ' . $user_id . ' for PCC UUID: ' . $pcc_uuid, 'INFO');
        
        mysqli_close($con);
        return $user_id;
        
    } catch (Exception $e) {
        // Rollback on error
        mysqli_rollback($con);
        pcc_log('Error creating DDxRx user: ' . $e->getMessage(), 'ERROR');
        mysqli_close($con);
        return false;
    }
}

/**
 * Update existing DDxRx user with PCC data
 * 
 * @param int $user_id DDxRx user ID
 * @param array $pcc_user_data User data from PCC API
 * @return bool Success status
 */
function pcc_update_ddxrx_user($user_id, $pcc_user_data) {
    pcc_log('Updating DDxRx user ID: ' . $user_id, 'DEBUG');
    
    $con = pcc_db_connect();
    
    if (!$con) {
        return false;
    }
    
    // Extract updated information
    $email = isset($pcc_user_data['email']) ? $pcc_user_data['email'] : '';
    $updated_at = date('Y-m-d H:i:s');
    
    // Update email if changed
    if (!empty($email)) {
        $stmt = mysqli_prepare($con, "UPDATE all_users SET email_address = ?, updated_at = ? WHERE userid = ?");
        
        if ($stmt) {
            mysqli_stmt_bind_param($stmt, "ssi", $email, $updated_at, $user_id);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            
            pcc_log('Updated user email for ID: ' . $user_id, 'DEBUG');
        }
    }
    
    mysqli_close($con);
    return true;
}

/**
 * Create DDxRx session for logged-in user
 * 
 * @param int $user_id DDxRx user ID
 * @param string $user_type DDxRx user type
 * @return bool Success status
 */
function pcc_create_session($user_id, $user_type) {
    pcc_log('Creating session for user ID: ' . $user_id . ', type: ' . $user_type, 'DEBUG');
    
    // Map user type to session user type
    $session_user_type = '';
    
    if ($user_type == '0') {
        $session_user_type = 'super_admin';
    } elseif ($user_type == '1') {
        $session_user_type = 'admin';
    } elseif ($user_type == '2') {
        $session_user_type = 'physician';
    } elseif ($user_type == '3') {
        $session_user_type = 'patient';
    } elseif ($user_type == '4') {
        $session_user_type = 'nurse';
    }
    
    // Set session variables (matching existing DDxRx session structure)
    $_SESSION['userid'] = $user_id;
    $_SESSION['userType'] = $session_user_type;
    $_SESSION['user_type'] = $session_user_type;
    $_SESSION['pcc_login'] = true;
    $_SESSION['pcc_login_time'] = time();
    
    pcc_log('Session created successfully for user ID: ' . $user_id, 'INFO');
    
    return true;
}

/**
 * Get dashboard redirect URL based on user type
 * 
 * @param string $user_type DDxRx user type
 * @return string Dashboard URL
 */
function pcc_get_dashboard_url($user_type) {
    // Determine redirect URL based on user type
    if ($user_type == '0' || $user_type == '1') {
        // Admin users
        return '/admin/index.php';
    } elseif ($user_type == '2') {
        // Physician users
        return '/physician_default.php';
    } elseif ($user_type == '4') {
        // Nurse users
        return '/nurse_default.php';
    } else {
        // Patient users (default)
        return '/default.php';
    }
}

/**
 * Validate PCC OAuth state parameter (CSRF protection)
 * 
 * @param string $state State parameter from PCC
 * @return bool Valid or not
 */
function pcc_validate_state($state) {
    if (!PCC_ENABLE_CSRF) {
        return true;
    }
    
    if (!isset($_SESSION['pcc_oauth_state'])) {
        pcc_log('No OAuth state found in session', 'WARNING');
        return false;
    }
    
    $is_valid = $_SESSION['pcc_oauth_state'] === $state;
    
    if (!$is_valid) {
        pcc_log('OAuth state mismatch - possible CSRF attack', 'ERROR');
    }
    
    // Clear the state after validation
    unset($_SESSION['pcc_oauth_state']);
    
    return $is_valid;
}

/**
 * Generate OAuth state parameter for CSRF protection
 * 
 * @return string State parameter
 */
function pcc_generate_state() {
    $state = bin2hex(openssl_random_pseudo_bytes(16));
    $_SESSION['pcc_oauth_state'] = $state;
    
    return $state;
}

?>
