<?php
/**
 * PointClickCare OAuth Callback Handler
 * 
 * This file handles the OAuth callback from PCC, exchanges the authorization
 * code for an access token, fetches user data, creates or updates the DDxRx
 * user account, and logs the user into the appropriate dashboard.
 * 
 * PHP 5.6/8.1 Compatible
 */

// Prevent direct access without authorization code
if (!isset($_GET['code'])) {
    header('HTTP/1.1 400 Bad Request');
    die('Error: No authorization code provided');
}

// Load configuration and helper functions
require_once(dirname(__FILE__) . '/pcc-config.php');
require_once(dirname(__FILE__) . '/includes/pcc-helper.php');

// Start processing
pcc_log('=== PCC OAuth Callback Started ===', 'INFO');
pcc_log('Request URI: ' . $_SERVER['REQUEST_URI'], 'DEBUG');

// ========================================
// STEP 1: Validate Request
// ========================================

$authorization_code = $_GET['code'];
$state = isset($_GET['state']) ? $_GET['state'] : '';
$error = isset($_GET['error']) ? $_GET['error'] : '';
$error_description = isset($_GET['error_description']) ? $_GET['error_description'] : '';

// Check for OAuth errors
if (!empty($error)) {
    pcc_log('OAuth error received: ' . $error . ' - ' . $error_description, 'ERROR');
    
    // Redirect to error page
    header('Location: /login.php?error=pcc_oauth_failed&message=' . urlencode($error_description));
    exit;
}

// Validate state parameter (CSRF protection)
if (PCC_ENABLE_CSRF) {
    if (!pcc_validate_state($state)) {
        pcc_log('Invalid OAuth state parameter - possible CSRF attack', 'ERROR');
        
        header('Location: /login.php?error=invalid_state');
        exit;
    }
}

pcc_log('Authorization code received: ' . substr($authorization_code, 0, 10) . '...', 'DEBUG');

// ========================================
// STEP 2: Exchange Code for Access Token
// ========================================

$token_data = pcc_exchange_code_for_token($authorization_code);

if (!$token_data) {
    pcc_log('Failed to exchange authorization code for access token', 'ERROR');
    
    header('Location: /login.php?error=token_exchange_failed');
    exit;
}

$access_token = $token_data['access_token'];
$refresh_token = isset($token_data['refresh_token']) ? $token_data['refresh_token'] : '';
$expires_in = isset($token_data['expires_in']) ? $token_data['expires_in'] : 3600;

pcc_log('Access token obtained successfully, expires in: ' . $expires_in . ' seconds', 'INFO');

// Store token in session for potential future use
$_SESSION['pcc_access_token'] = $access_token;
$_SESSION['pcc_refresh_token'] = $refresh_token;
$_SESSION['pcc_token_expires'] = time() + $expires_in;

// ========================================
// STEP 3: Fetch User Data from PCC API
// ========================================

$pcc_user_data = pcc_get_user_data($access_token, $token_data);

if (!$pcc_user_data) {
    pcc_log('Failed to fetch user data from PCC API', 'ERROR');
    
    header('Location: /login.php?error=user_data_failed');
    exit;
}

pcc_log('User data fetched successfully', 'INFO');

// Extract critical user information
// Note: PCC API userinfo endpoint structure may differ from full user API
$pcc_uuid = '';
$pcc_username = '';
$pcc_email = '';
$pcc_first_name = '';
$pcc_last_name = '';
$pcc_role = 'testuser'; // Default role if not found
$org_uuid = '';

// Try to get user ID from userinfo API response
if (isset($pcc_user_data['id'])) {
    $pcc_uuid = $pcc_user_data['id'];
} elseif (isset($pcc_user_data['userId'])) {
    $pcc_uuid = $pcc_user_data['userId'];
}

// Try to get username
if (isset($pcc_user_data['username'])) {
    $pcc_username = $pcc_user_data['username'];
} elseif (isset($pcc_user_data['displayName'])) {
    $pcc_username = $pcc_user_data['displayName'];
}

// Try to get email
if (isset($pcc_user_data['email'])) {
    $pcc_email = $pcc_user_data['email'];
}

// Try to get names
if (isset($pcc_user_data['firstName'])) {
    $pcc_first_name = $pcc_user_data['firstName'];
} elseif (isset($pcc_user_data['givenName'])) {
    $pcc_first_name = $pcc_user_data['givenName'];
}

if (isset($pcc_user_data['lastName'])) {
    $pcc_last_name = $pcc_user_data['lastName'];
} elseif (isset($pcc_user_data['familyName'])) {
    $pcc_last_name = $pcc_user_data['familyName'];
}

// Try to get role/privileges
if (isset($pcc_user_data['role'])) {
    $pcc_role = $pcc_user_data['role'];
} elseif (isset($pcc_user_data['roles']) && is_array($pcc_user_data['roles']) && count($pcc_user_data['roles']) > 0) {
    $pcc_role = $pcc_user_data['roles'][0]; // Use first role
} elseif (isset($pcc_user_data['privileges']) && is_array($pcc_user_data['privileges']) && count($pcc_user_data['privileges']) > 0) {
    $pcc_role = $pcc_user_data['privileges'][0]; // Use first privilege
}

// Get org UUID from token data if not in user data
if (isset($pcc_user_data['orgUuid'])) {
    $org_uuid = $pcc_user_data['orgUuid'];
} elseif (isset($token_data['metadata']['orgUuid'])) {
    $org_uuid = $token_data['metadata']['orgUuid'];
}

// If we still don't have a user UUID, try to generate one from available data
if (empty($pcc_uuid)) {
    // We might need to use a combination of username and org for unique identification
    if (!empty($pcc_username) && !empty($org_uuid)) {
        $pcc_uuid = $org_uuid . '.' . $pcc_username;
        pcc_log('Generated PCC UUID from username and org: ' . $pcc_uuid, 'INFO');
    } elseif (!empty($pcc_username)) {
        $pcc_uuid = 'pcc.' . $pcc_username;
        pcc_log('Generated PCC UUID from username only: ' . $pcc_uuid, 'INFO');
    }
}

// Validate required fields
if (empty($pcc_uuid)) {
    pcc_log('Missing required PCC user UUID', 'ERROR');
    
    header('Location: /login.php?error=missing_user_id');
    exit;
}

pcc_log('PCC User UUID: ' . $pcc_uuid . ', Username: ' . $pcc_username . ', Role: ' . $pcc_role, 'DEBUG');

// ========================================
// STEP 4: Check if User Exists in DDxRx
// ========================================

$existing_user = pcc_check_user_exists($pcc_uuid);

if ($existing_user) {
    // User exists - log them in
    pcc_log('Existing user found - logging in', 'INFO');
    
    $user_id = $existing_user['userid'];
    $user_type = $existing_user['user_type'];
    
    // Optionally update user data if changed
    pcc_update_ddxrx_user($user_id, $pcc_user_data);
    
} else {
    // New user - create account
    pcc_log('New user - creating DDxRx account', 'INFO');
    
    $user_id = pcc_create_ddxrx_user($pcc_user_data);
    
    if (!$user_id) {
        pcc_log('Failed to create DDxRx user account', 'ERROR');
        
        header('Location: /login.php?error=account_creation_failed');
        exit;
    }
    
    // Get the user type that was assigned
    $user_type = pcc_map_role_to_usertype($pcc_role);
    
    pcc_log('New DDxRx user created with ID: ' . $user_id, 'INFO');
}

// ========================================
// STEP 5: Create DDxRx Session
// ========================================

$session_created = pcc_create_session($user_id, $user_type);

if (!$session_created) {
    pcc_log('Failed to create session for user ID: ' . $user_id, 'ERROR');
    
    header('Location: /login.php?error=session_failed');
    exit;
}

pcc_log('Session created successfully for user ID: ' . $user_id, 'INFO');

// ========================================
// STEP 6: Redirect to Appropriate Dashboard
// ========================================

$dashboard_url = pcc_get_dashboard_url($user_type);

pcc_log('Redirecting to dashboard: ' . $dashboard_url, 'INFO');
pcc_log('=== PCC OAuth Callback Completed Successfully ===', 'INFO');

// Redirect to dashboard
header('Location: ' . $dashboard_url);
exit;

?>
