<?php
/**
 * PointClickCare Integration Test Page
 * 
 * This page helps test and debug the PCC integration.
 * 
 * WARNING: Remove or protect this file in production!
 */

// Load configuration
require_once(dirname(__FILE__) . '/pcc-config.php');
require_once(dirname(__FILE__) . '/includes/pcc-helper.php');

// Check if we should run a specific test
$test = isset($_GET['test']) ? $_GET['test'] : '';
$output = array();

if ($test == 'db_connection') {
    // Test database connection
    $con = pcc_db_connect();
    if ($con) {
        $output['status'] = 'success';
        $output['message'] = 'Database connection successful';
        mysqli_close($con);
    } else {
        $output['status'] = 'error';
        $output['message'] = 'Database connection failed';
    }
    
} elseif ($test == 'check_column') {
    // Check if pcc_user_uuid column exists
    $con = pcc_db_connect();
    if ($con) {
        $result = mysqli_query($con, "SHOW COLUMNS FROM all_users LIKE 'pcc_user_uuid'");
        if (mysqli_num_rows($result) > 0) {
            $output['status'] = 'success';
            $output['message'] = 'pcc_user_uuid column exists';
        } else {
            $output['status'] = 'error';
            $output['message'] = 'pcc_user_uuid column NOT found - run database migration';
        }
        mysqli_close($con);
    } else {
        $output['status'] = 'error';
        $output['message'] = 'Database connection failed';
    }
    
} elseif ($test == 'role_mapping') {
    // Test role mapping
    $test_roles = array('physician', 'registerednurse', 'facilitymanager', 'testuser', 'unknown_role');
    $mappings = array();
    
    foreach ($test_roles as $role) {
        $user_type = pcc_map_role_to_usertype($role);
        $mappings[] = array(
            'role' => $role,
            'userType' => $user_type
        );
    }
    
    $output['status'] = 'success';
    $output['message'] = 'Role mapping test completed';
    $output['mappings'] = $mappings;
    
} elseif ($test == 'generate_password') {
    // Test password generation
    $password = pcc_generate_password(32);
    $output['status'] = 'success';
    $output['message'] = 'Password generated successfully';
    $output['password'] = $password;
    $output['length'] = strlen($password);
    
} elseif ($test == 'check_logs') {
    // Check if logs directory is writable
    $log_dir = dirname(PCC_LOG_FILE);
    $is_writable = is_writable($log_dir);
    
    if ($is_writable) {
        $output['status'] = 'success';
        $output['message'] = 'Logs directory is writable';
        $output['log_file'] = PCC_LOG_FILE;
    } else {
        $output['status'] = 'error';
        $output['message'] = 'Logs directory is NOT writable';
        $output['log_dir'] = $log_dir;
    }
    
} elseif ($test == 'oauth_urls') {
    // Display OAuth URLs for verification
    $output['status'] = 'success';
    $output['message'] = 'OAuth URLs';
    $output['urls'] = array(
        'initiate' => PCC_AUTHORIZE_URL,
        'callback' => PCC_OAUTH_CALLBACK_URL,
        'token_endpoint' => PCC_TOKEN_URL,
        'user_info_endpoint' => PCC_USER_INFO_URL
    );
}

// If a test was run, output JSON
if (!empty($test)) {
    header('Content-Type: application/json');
    echo json_encode($output, JSON_PRETTY_PRINT);
    exit;
}

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PCC Integration Test</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .header {
            background: #2c3e50;
            color: white;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .test-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        .test-card {
            background: white;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .test-card h3 {
            margin-top: 0;
            color: #2c3e50;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 3px;
            border: none;
            cursor: pointer;
        }
        .btn:hover {
            background: #2980b9;
        }
        .btn-success {
            background: #27ae60;
        }
        .btn-success:hover {
            background: #229954;
        }
        .btn-warning {
            background: #f39c12;
        }
        .btn-warning:hover {
            background: #e67e22;
        }
        .result {
            margin-top: 15px;
            padding: 10px;
            border-radius: 3px;
            display: none;
        }
        .result.success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        .result.error {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        .config-info {
            background: white;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .config-info h3 {
            margin-top: 0;
            color: #2c3e50;
        }
        .config-item {
            display: flex;
            padding: 8px 0;
            border-bottom: 1px solid #ecf0f1;
        }
        .config-label {
            flex: 0 0 200px;
            font-weight: bold;
            color: #7f8c8d;
        }
        .config-value {
            flex: 1;
            font-family: monospace;
            color: #2c3e50;
        }
        pre {
            background: #2c3e50;
            color: #ecf0f1;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🔧 PCC Integration Test Suite</h1>
        <p>Test and verify your PointClickCare integration setup</p>
    </div>

    <div class="test-grid">
        <!-- Database Connection Test -->
        <div class="test-card">
            <h3>Database Connection</h3>
            <p>Test connection to DDxRx database</p>
            <button class="btn" onclick="runTest('db_connection', this)">Run Test</button>
            <div class="result" id="result-db_connection"></div>
        </div>

        <!-- Column Check Test -->
        <div class="test-card">
            <h3>Database Migration</h3>
            <p>Check if pcc_user_uuid column exists</p>
            <button class="btn" onclick="runTest('check_column', this)">Run Test</button>
            <div class="result" id="result-check_column"></div>
        </div>

        <!-- Role Mapping Test -->
        <div class="test-card">
            <h3>Role Mapping</h3>
            <p>Test PCC role to DDxRx userType mapping</p>
            <button class="btn" onclick="runTest('role_mapping', this)">Run Test</button>
            <div class="result" id="result-role_mapping"></div>
        </div>

        <!-- Password Generation Test -->
        <div class="test-card">
            <h3>Password Generation</h3>
            <p>Test secure password generation</p>
            <button class="btn" onclick="runTest('generate_password', this)">Run Test</button>
            <div class="result" id="result-generate_password"></div>
        </div>

        <!-- Logs Check Test -->
        <div class="test-card">
            <h3>Logging System</h3>
            <p>Check if logs directory is writable</p>
            <button class="btn" onclick="runTest('check_logs', this)">Run Test</button>
            <div class="result" id="result-check_logs"></div>
        </div>

        <!-- OAuth URLs Test -->
        <div class="test-card">
            <h3>OAuth Configuration</h3>
            <p>Display configured OAuth endpoints</p>
            <button class="btn" onclick="runTest('oauth_urls', this)">Run Test</button>
            <div class="result" id="result-oauth_urls"></div>
        </div>
    </div>

    <!-- Configuration Info -->
    <div class="config-info">
        <h3>Current Configuration</h3>
        <div class="config-item">
            <div class="config-label">Environment:</div>
            <div class="config-value"><?php echo PCC_ENVIRONMENT; ?></div>
        </div>
        <div class="config-item">
            <div class="config-label">Customer Key:</div>
            <div class="config-value"><?php echo substr(PCC_CUSTOMER_KEY, 0, 10); ?>...</div>
        </div>
        <div class="config-item">
            <div class="config-label">Database:</div>
            <div class="config-value"><?php echo PCC_DB_NAME; ?></div>
        </div>
        <div class="config-item">
            <div class="config-label">Callback URL:</div>
            <div class="config-value"><?php echo PCC_OAUTH_CALLBACK_URL; ?></div>
        </div>
        <div class="config-item">
            <div class="config-label">Default Group ID:</div>
            <div class="config-value"><?php echo PCC_DEFAULT_GROUP_ID; ?></div>
        </div>
        <div class="config-item">
            <div class="config-label">Logging Enabled:</div>
            <div class="config-value"><?php echo PCC_ENABLE_LOGGING ? 'Yes' : 'No'; ?></div>
        </div>
        <div class="config-item">
            <div class="config-label">CSRF Protection:</div>
            <div class="config-value"><?php echo PCC_ENABLE_CSRF ? 'Enabled' : 'Disabled'; ?></div>
        </div>
    </div>

    <!-- OAuth Flow Links -->
    <div class="config-info" style="margin-top: 20px;">
        <h3>OAuth Flow</h3>
        <p><a href="oauth-initiate.php" class="btn btn-success">Start OAuth Flow</a></p>
        <p style="color: #7f8c8d; font-size: 14px;">
            Click above to start the OAuth flow and test the complete integration.
            You'll be redirected to PCC login page.
        </p>
    </div>

    <!-- View Logs -->
    <div class="config-info" style="margin-top: 20px;">
        <h3>Recent Logs</h3>
        <?php
        if (file_exists(PCC_LOG_FILE)) {
            $logs = file(PCC_LOG_FILE);
            $recent_logs = array_slice($logs, -20);
            echo '<pre>' . htmlspecialchars(implode('', $recent_logs)) . '</pre>';
        } else {
            echo '<p>No log file found. Logs will be created when integration is used.</p>';
        }
        ?>
    </div>

    <script>
        function runTest(testName, button) {
            // Disable button
            button.disabled = true;
            button.textContent = 'Running...';
            
            // Hide previous result
            const resultDiv = document.getElementById('result-' + testName);
            resultDiv.style.display = 'none';
            
            // Run test
            fetch('?test=' + testName)
                .then(response => response.json())
                .then(data => {
                    // Show result
                    resultDiv.className = 'result ' + data.status;
                    resultDiv.style.display = 'block';
                    
                    let html = '<strong>' + data.message + '</strong>';
                    
                    // Add additional data if present
                    if (data.mappings) {
                        html += '<ul>';
                        data.mappings.forEach(m => {
                            html += '<li>' + m.role + ' → userType ' + m.userType + '</li>';
                        });
                        html += '</ul>';
                    }
                    
                    if (data.password) {
                        html += '<p>Generated: <code>' + data.password + '</code></p>';
                        html += '<p>Length: ' + data.length + ' characters</p>';
                    }
                    
                    if (data.urls) {
                        html += '<ul>';
                        for (let key in data.urls) {
                            html += '<li><strong>' + key + ':</strong><br><small>' + data.urls[key] + '</small></li>';
                        }
                        html += '</ul>';
                    }
                    
                    if (data.log_file) {
                        html += '<p>Log file: <code>' + data.log_file + '</code></p>';
                    }
                    
                    if (data.log_dir) {
                        html += '<p>Directory: <code>' + data.log_dir + '</code></p>';
                    }
                    
                    resultDiv.innerHTML = html;
                    
                    // Re-enable button
                    button.disabled = false;
                    button.textContent = 'Run Test';
                })
                .catch(error => {
                    resultDiv.className = 'result error';
                    resultDiv.style.display = 'block';
                    resultDiv.innerHTML = '<strong>Error:</strong> ' + error;
                    
                    button.disabled = false;
                    button.textContent = 'Run Test';
                });
        }
    </script>
</body>
</html>
