<?php

/**
 * Simple File-Based Cache for SQL Query Results
 * 
 * This class provides a lightweight caching mechanism specifically designed
 * for caching SQL query results without external dependencies.
 * Compatible with PHP 5.6+
 */
class SimpleCache {
    
    private $cache_dir;
    private $default_ttl;
    
    /**
     * Constructor
     * 
     * @param string $cache_dir Directory to store cache files
     * @param int $default_ttl Default time-to-live in seconds (default: 300 = 5 minutes)
     */
    public function __construct($cache_dir = null, $default_ttl = 300) {
        $this->cache_dir = $cache_dir ? $cache_dir : dirname(__FILE__) . '/cache';
        $this->default_ttl = $default_ttl;
        
        // Create cache directory if it doesn't exist
        if (!is_dir($this->cache_dir)) {
            if (!mkdir($this->cache_dir, 0755, true)) {
                throw new Exception("Cannot create cache directory: " . $this->cache_dir);
            }
        }
        
        // Create .htaccess to protect cache directory
        $htaccess_file = $this->cache_dir . '/.htaccess';
        if (!file_exists($htaccess_file)) {
            file_put_contents($htaccess_file, "Deny from all\n");
        }
    }
    
    /**
     * Generate cache key from string
     * 
     * @param string $key The cache key
     * @return string Sanitized cache key
     */
    private function getCacheKey($key) {
        return md5($key);
    }
    
    /**
     * Get cache file path
     * 
     * @param string $key Cache key
     * @return string Full path to cache file
     */
    private function getCacheFilePath($key) {
        $cache_key = $this->getCacheKey($key);
        return $this->cache_dir . '/' . $cache_key . '.cache';
    }
    
    /**
     * Store data in cache
     * 
     * @param string $key Cache key
     * @param mixed $data Data to cache
     * @param int $ttl Time-to-live in seconds (optional, uses default if not specified)
     * @return bool True on success, false on failure
     */
    public function set($key, $data, $ttl = null) {
        if ($ttl === null) {
            $ttl = $this->default_ttl;
        }
        
        $cache_file = $this->getCacheFilePath($key);
        $cache_data = array(
            'created' => time(),
            'ttl' => $ttl,
            'data' => $data
        );
        
        $serialized_data = serialize($cache_data);
        return file_put_contents($cache_file, $serialized_data, LOCK_EX) !== false;
    }
    
    /**
     * Retrieve data from cache
     * 
     * @param string $key Cache key
     * @param mixed $default Default value to return if cache miss
     * @return mixed Cached data or default value
     */
    public function get($key, $default = null) {
        $cache_file = $this->getCacheFilePath($key);
        
        if (!file_exists($cache_file)) {
            return $default;
        }
        
        $cache_data = unserialize(file_get_contents($cache_file));
        
        if (!$cache_data || !is_array($cache_data)) {
            return $default;
        }
        
        // Check if cache has expired
        $age = time() - $cache_data['created'];
        if ($age > $cache_data['ttl']) {
            $this->delete($key);
            return $default;
        }
        
        return $cache_data['data'];
    }
    
    /**
     * Check if cache key exists and is valid
     * 
     * @param string $key Cache key
     * @return bool True if exists and valid, false otherwise
     */
    public function has($key) {
        return $this->get($key, '__CACHE_MISS__') !== '__CACHE_MISS__';
    }
    
    /**
     * Delete cache entry
     * 
     * @param string $key Cache key
     * @return bool True on success, false on failure
     */
    public function delete($key) {
        $cache_file = $this->getCacheFilePath($key);
        if (file_exists($cache_file)) {
            return unlink($cache_file);
        }
        return true;
    }
    
    /**
     * Clear all cache entries
     * 
     * @return bool True on success, false on failure
     */
    public function clear() {
        $files = glob($this->cache_dir . '/*.cache');
        $success = true;
        
        foreach ($files as $file) {
            if (!unlink($file)) {
                $success = false;
            }
        }
        
        return $success;
    }
    
    /**
     * Clear cache entries matching a pattern
     * 
     * @param string $pattern Pattern to match in cache keys
     * @return bool True on success, false on failure
     */
    public function clearByPattern($pattern) {
        $files = glob($this->cache_dir . '/*.cache');
        $success = true;
        
        foreach ($files as $file) {
            // For pattern matching, we need to check if any cached keys match the pattern
            // Since we're using MD5 hashes as filenames, we'll check all files and
            // clear them if the pattern matches the original key pattern
            if ($pattern === '*') {
                // Clear all files
                if (!unlink($file)) {
                    $success = false;
                }
            } else {
                // For our specific use case, clear all files that start with the pattern
                // We'll create a hash of the pattern and check if any files might match
                $pattern_variations = array(
                    $this->getCacheKey($pattern . '_doctor'),
                    $this->getCacheKey($pattern . '_nurse'),
                    $this->getCacheKey($pattern . '_confirmed'),
                    $this->getCacheKey($pattern . '_waiting'),
                    $this->getCacheKey($pattern . '_all')
                );
                
                $filename = basename($file, '.cache');
                
                // Check if filename starts with any pattern variation
                $should_delete = false;
                foreach ($pattern_variations as $pattern_hash) {
                    if (strpos($pattern_hash, substr($filename, 0, 8)) === 0 || 
                        strpos($filename, substr($pattern_hash, 0, 8)) === 0) {
                        $should_delete = true;
                        break;
                    }
                }
                
                // For our confirmed_diagnosis pattern, we'll be more aggressive and clear
                // any cache file that could be related to waiting room data
                if (strpos($pattern, 'confirmed_diagnosis') !== false) {
                    $should_delete = true;
                }
                
                if ($should_delete && !unlink($file)) {
                    $success = false;
                }
            }
        }
        
        return $success;
    }
    
    /**
     * Get cache statistics
     * 
     * @return array Array with cache statistics
     */
    public function getStats() {
        $files = glob($this->cache_dir . '/*.cache');
        $total_files = count($files);
        $total_size = 0;
        $expired_files = 0;
        
        foreach ($files as $file) {
            $size = filesize($file);
            $total_size += $size;
            
            $cache_data = unserialize(file_get_contents($file));
            if ($cache_data && is_array($cache_data)) {
                $age = time() - $cache_data['created'];
                if ($age > $cache_data['ttl']) {
                    $expired_files++;
                }
            }
        }
        
        return array(
            'total_files' => $total_files,
            'total_size_bytes' => $total_size,
            'total_size_mb' => round($total_size / 1048576, 2),
            'expired_files' => $expired_files,
            'cache_directory' => $this->cache_dir
        );
    }
    
    /**
     * Clean up expired cache entries
     * 
     * @return int Number of files cleaned up
     */
    public function cleanup() {
        $files = glob($this->cache_dir . '/*.cache');
        $cleaned_count = 0;
        
        foreach ($files as $file) {
            $cache_data = unserialize(file_get_contents($file));
            if ($cache_data && is_array($cache_data)) {
                $age = time() - $cache_data['created'];
                if ($age > $cache_data['ttl']) {
                    if (unlink($file)) {
                        $cleaned_count++;
                    }
                }
            }
        }
        
        return $cleaned_count;
    }
}

?>