$(document).ready(function() {
    
    // Autocomplete functionality for symptom search
    $('#search-input').keyup(function(){
        var search_qry = $(this).val().trim();
        var symptomList = '';

        if (search_qry.length >= 2) {
            // Start Ajax
            $.ajax({
                type: 'POST',
                url: 'process/search-symptoms.php',
                data: {
                    search_qry: search_qry
                },
                dataType: 'json',
                beforeSend: function() {
                    $('.search-result-placeholder').hide();
                },
                success: function(response) {
                    if(response.result == true){
                        // Build the symptom list
                        response.items.forEach(function(item) {
                            symptomList += '<li data-symptom="' + item.symptom_name + '">' + item.symptom_name + '</li>';
                        });

                        // Update the search result placeholder
                        $('.search-result-placeholder').html(symptomList);
                        $('.search-result-placeholder').show();
                    } else {
                        $('.search-result-placeholder').hide();
                    }
                },
                error: function() {
                    $('.search-result-placeholder').hide();
                }
            });
        } else {
            $('.search-result-placeholder').html('');
            $('.search-result-placeholder').hide();
        }
    });

    // Handle autocomplete item selection
    $(document).on('click', '.search-result-placeholder li', function(){
        var symptomName = $(this).data('symptom');
        var form = $('#search-form');
        var existingInputs = form.find('input[name^="symptoms"]');
        
        // Check if symptom already exists
        var symptomExists = false;
        existingInputs.each(function() {
            if ($(this).val().toLowerCase() === symptomName.toLowerCase()) {
                symptomExists = true;
                return false;
            }
        });
        
        if (!symptomExists) {
            var newIndex = existingInputs.length;
            
            // Add the new symptom to form
            form.append('<input type="hidden" name="symptoms[' + newIndex + ']" value="' + symptomName + '">');
            
            // Create and add new symptom element with smooth animation
            var newSymptomElement = $('<a href="#" data-item="' + symptomName + '" style="display: none;"><span>' + symptomName + '</span><span class="material-symbols-outlined icon"> cancel </span></a>');
            $('.symptom-list div').append(newSymptomElement);
            newSymptomElement.fadeIn(300);
            
            // Show success message
            Swal.fire({
                position: "bottom-end",
                icon: "success",
                title: symptomName + " added to symptoms",
                showConfirmButton: false,
                toast: true,
                timer: 2500
            });
        } else {
            // Show already exists message
            Swal.fire({
                position: "bottom-end",
                icon: "info",
                title: symptomName + " already added",
                showConfirmButton: false,
                toast: true,
                timer: 2000
            });
        }
        
        // Clear the input field and hide results
        $('#search-input').val('');
        $('.search-result-placeholder').hide();
    });

    // Hide autocomplete when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('#search-form').length) {
            $('.search-result-placeholder').hide();
        }
    });

    // Hide autocomplete on escape key
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape') {
            $('.search-result-placeholder').hide();
        }
    });
    
    // Handle symptom removal from the list (using event delegation for dynamic elements)
    $(document).on('click', '.symptom-list a', function(e) {
        e.preventDefault();
        e.stopPropagation(); // Prevent any parent form submission
        
        var symptomToRemove = $(this).data('item');
        var $symptomElement = $(this);
        
        // Add smooth fade out animation before removal
        $symptomElement.fadeOut(300, function() {
            // Remove from form
            var form = $('#search-form');
            var inputs = form.find('input[name^="symptoms"]');
            var newSymptoms = [];
            
            inputs.each(function() {
                if ($(this).val() !== symptomToRemove) {
                    newSymptoms.push($(this).val());
                }
            });
            
            // Clear existing hidden inputs
            inputs.remove();
            
            // Add remaining symptoms back to form with proper indexing
            for (var i = 0; i < newSymptoms.length; i++) {
                form.append('<input type="hidden" name="symptoms[' + i + ']" value="' + newSymptoms[i] + '">');
            }
            
            // Remove the element from DOM
            $symptomElement.remove();
            
            // Update any "Already Added" buttons back to "Add Symptom" state
            $('.add_symptom.already-added').each(function() {
                var buttonSymptom = $(this).closest('.checkbox-label').find('h6').text().trim();
                if (buttonSymptom.toLowerCase() === symptomToRemove.toLowerCase()) {
                    $(this).removeClass('already-added');
                    $(this).find('.material-symbols-outlined').text('add_circle');
                    $(this).contents().filter(function() { return this.nodeType === 3; }).last().replaceWith(' Add Symptom');
                }
            });
        });
        
        // DO NOT submit form - just update the UI
    });
    
    // Handle adding symptoms from questions
    $('.add_symptom').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation(); // Prevent any parent form submission
        
        if ($(this).hasClass('already-added')) {
            return;
        }
        
        var questionTitle = $(this).closest('.checkbox-label').find('h6').text().trim();
        var form = $('#search-form');
        var existingInputs = form.find('input[name^="symptoms"]');
        var newIndex = existingInputs.length;
        
        // Check if symptom already exists
        var symptomExists = false;
        existingInputs.each(function() {
            if ($(this).val().toLowerCase() === questionTitle.toLowerCase()) {
                symptomExists = true;
                return false;
            }
        });
        
        if (!symptomExists) {
            // Add the new symptom to form
            form.append('<input type="hidden" name="symptoms[' + newIndex + ']" value="' + questionTitle + '">');
            
            // Create and add new symptom element with smooth animation
            var newSymptomElement = $('<a href="#" data-item="' + questionTitle + '" style="display: none;"><span>' + questionTitle + '</span><span class="material-symbols-outlined icon"> cancel </span></a>');
            $('.symptom-list div').append(newSymptomElement);
            newSymptomElement.fadeIn(300);
            
            // Update button state
            $(this).addClass('already-added');
            $(this).find('.material-symbols-outlined').text('check_circle');
            $(this).contents().filter(function() { return this.nodeType === 3; }).last().replaceWith(' Already Added');
        }
        
        // DO NOT submit form - just update the UI
    });
    
    // Handle checkbox toggling
    $('.checkbox-toggle').on('click', function(e) {
        e.stopPropagation(); // Prevent any parent form submission
        
        var $item = $(this).closest('.item');
        var $icon = $(this).find('.icon');
        var $hiddenInput = $item.find('input[type="hidden"]');
        var dataYes = $item.data('yes');
        var dataNo = $item.data('no');
        
        if ($icon.hasClass('add')) {
            // Change to checked state with smooth animation
            $icon.removeClass('add').addClass('added').text('check_box');
            $hiddenInput.val(dataYes);
            
            // Add smooth fade animation
            $item.addClass('selected').hide().fadeIn(300);
        } else {
            // Change to unchecked state with smooth animation
            $icon.removeClass('added').addClass('add').text('check_box_outline_blank');
            $hiddenInput.val(dataNo);
            
            // Add smooth fade animation
            $item.removeClass('selected').fadeOut(150).fadeIn(300);
        }
    });
    
    // Handle form submission
    $('form[action="process/diagnosis-question-result.php"]').on('submit', function(e) {
        e.preventDefault();
        
        var $form = $(this);
        var $submitButton = $form.find('input[type="submit"]');
        var originalText = $submitButton.val();
        
        // Get waitingroom_id from localStorage
        var patientData = localStorage.getItem('patient');
        var waitingroom_id = 0;
        
        if (patientData) {
            var patient = JSON.parse(patientData);
            waitingroom_id = patient.waitingroom_id;
        }
        
        if (waitingroom_id == 0) {
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Unable to find patient waiting room information.',
                allowOutsideClick: false,
                allowEscapeKey: false
            });
            return;
        }
        
        // Get original symptoms from the symptom list
        var originalSymptoms = [];
        $('.symptom-list [data-item]').each(function() {
            originalSymptoms.push($(this).data('item'));
        });
        
        // Add waitingroom_id and original symptoms to form data
        var formData = $form.serialize() + '&waitingroom_id=' + waitingroom_id + '&original_symptoms=' + encodeURIComponent(originalSymptoms.join(', '));
        
        // Disable submit button
        $submitButton.prop('disabled', true).val('Processing...');
        
        // Show loading if available
        if ($('.loading-screen').length > 0) {
            $('.loading-screen').fadeIn();
        }
        
        $.ajax({
            url: 'process/diagnosis-question-result.php',
            type: 'POST',
            data: formData,
            dataType: 'json',
            success: function(response) {
                if ($('.loading-screen').length > 0) {
                    $('.loading-screen').fadeOut();
                }
                $submitButton.prop('disabled', false).val(originalText);
                
                if (response.success) {
                    if (response.diagnosis_confirmed) {
                        // Diagnosis confirmed - emphasize "View Results" button
                        Swal.fire({
                            icon: 'success',
                            title: 'Diagnosis Confirmed!',
                            text: response.message,
                            allowOutsideClick: false,
                            allowEscapeKey: false,
                            showCancelButton: true,
                            confirmButtonText: 'View Results',
                            cancelButtonText: 'More Diagnoses',
                            buttonsStyling: false,
                            customClass: {
                                confirmButton: 'btn btn-success btn-lg swal2-confirm-primary',
                                cancelButton: 'btn btn-outline-secondary swal2-cancel-secondary'
                            },
                            reverseButtons: false
                        }).then((result) => {
                            if (result.isConfirmed) {
                                // View Results - redirect based on user type
                                window.location.href = getResultsUrl();
                            } else if (result.dismiss === Swal.DismissReason.cancel) {
                                // More diagnoses
                                showNextDiagnosis();
                            }
                        });
                    } else {
                        // Diagnosis not confirmed - show "Show Next Diagnoses" button
                        Swal.fire({
                            icon: 'info',
                            title: 'Criteria Not Met',
                            text: response.message,
                            allowOutsideClick: false,
                            allowEscapeKey: false,
                            confirmButtonText: 'Show Next Diagnoses',
                            buttonsStyling: false,
                            customClass: {
                                confirmButton: 'btn btn-primary'
                            }
                        }).then((result) => {
                            if (result.isConfirmed) {
                                showNextDiagnosis();
                            }
                        });
                    }
                } else {
                    // Error handling with skip option for missing criteria
                    if (response.message && response.message.includes('Decision support criteria not found')) {
                        Swal.fire({
                            icon: 'warning',
                            title: 'No Criteria Found',
                            text: response.message,
                            allowOutsideClick: false,
                            allowEscapeKey: false,
                            showCancelButton: true,
                            confirmButtonText: 'Skip to Next Diagnosis',
                            cancelButtonText: 'Cancel',
                            buttonsStyling: false,
                            customClass: {
                                confirmButton: 'btn btn-warning',
                                cancelButton: 'btn btn-outline-secondary'
                            }
                        }).then((result) => {
                            if (result.isConfirmed) {
                                showNextDiagnosis();
                            }
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: response.error || response.message || 'An error occurred while processing the diagnosis.',
                            allowOutsideClick: false,
                            allowEscapeKey: false,
                            confirmButtonText: 'OK',
                            buttonsStyling: false,
                            customClass: {
                                confirmButton: 'btn btn-danger'
                            }
                        });
                    }
                }
            },
            error: function(xhr, status, error) {
                if ($('.loading-screen').length > 0) {
                    $('.loading-screen').fadeOut();
                }
                $submitButton.prop('disabled', false).val(originalText);
                
                Swal.fire({
                    icon: 'error',
                    title: 'Network Error',
                    text: 'Unable to process diagnosis. Please try again.',
                    allowOutsideClick: false,
                    allowEscapeKey: false,
                    confirmButtonText: 'OK',
                    buttonsStyling: false,
                    customClass: {
                        confirmButton: 'btn btn-danger'
                    }
                });
            }
        });
    });
    
    // Handle skip diagnosis button (for diagnoses without questions)
    $(document).on('click', '#skip-diagnosis', function(e) {
        e.preventDefault();
        
        Swal.fire({
            icon: 'question',
            title: 'Skip Diagnosis',
            text: 'This diagnosis has no questions to answer. Would you like to skip to the next diagnosis?',
            allowOutsideClick: false,
            allowEscapeKey: false,
            showCancelButton: true,
            confirmButtonText: 'Skip to Next',
            cancelButtonText: 'Cancel',
            buttonsStyling: false,
            customClass: {
                confirmButton: 'btn btn-warning',
                cancelButton: 'btn btn-outline-secondary'
            }
        }).then((result) => {
            if (result.isConfirmed) {
                showNextDiagnosis();
            }
        });
    });
    
    // Handle reset questions button
    $(document).on('click', '#reset-questions', function(e) {
        e.preventDefault();
        
        Swal.fire({
            icon: 'question',
            title: 'Reset Questions',
            text: 'This will reset all answers for the current diagnosis to default values (matching selected symptoms). Continue?',
            allowOutsideClick: false,
            allowEscapeKey: false,
            showCancelButton: true,
            confirmButtonText: 'Reset',
            cancelButtonText: 'Cancel',
            buttonsStyling: false,
            customClass: {
                confirmButton: 'btn btn-warning',
                cancelButton: 'btn btn-outline-secondary'
            }
        }).then((result) => {
            if (result.isConfirmed) {
                resetCurrentDiagnosisQuestions();
            }
        });
    });
    
    
    // Function to reset questions to default state
    function resetCurrentDiagnosisQuestions() {
        // Get current diagnosis container
        var $currentDiagnosis = $('.diagnosed-item:not(.hidden-diagnosis)').first();
        
        // Remove selected class from all items in the current diagnosis
        $currentDiagnosis.find('.questions .item').removeClass('selected');

        // Get selected symptoms from the symptom list
        var selectedSymptoms = [];
        $('.symptom-list [data-item]').each(function() {
            selectedSymptoms.push($(this).data('item').toLowerCase());
        });
        
        // Reset all checkboxes and their hidden inputs
        $currentDiagnosis.find('.item').each(function() {
            var $item = $(this);
            var $icon = $item.find('.checkbox-toggle .icon');
            var $hiddenInput = $item.find('input[type="hidden"]');
            var questionText = $item.find('h6').text().toLowerCase();
            var dataYes = $item.data('yes');
            var dataNo = $item.data('no');
            var shouldCheck = false;
            
            // Check if any selected symptom matches this question
            for (var i = 0; i < selectedSymptoms.length; i++) {
                if (questionText.indexOf(selectedSymptoms[i]) !== -1 || selectedSymptoms[i].indexOf(questionText) !== -1) {
                    shouldCheck = true;
                    break;
                }
            }
            
            if (shouldCheck) {
                // Set to checked state
                $icon.removeClass('add').addClass('added').text('check_box');
                $hiddenInput.val(dataYes);
            } else {
                // Set to unchecked state
                $icon.removeClass('added').addClass('add').text('check_box_outline_blank');
                $hiddenInput.val(dataNo);
            }
        });
        
        // Show success message
        Swal.fire({
            icon: 'success',
            title: 'Questions Reset',
            text: 'All questions have been reset to match your selected symptoms.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            timer: 2000,
            timerProgressBar: true,
            showConfirmButton: false
        });
    }
    
    // Function to show next diagnosis
    function showNextDiagnosis() {
        var $currentDiagnosis = $('.diagnosed-item:not(.hidden-diagnosis)');
        var $nextDiagnosis = $currentDiagnosis.next('.diagnosed-item.hidden-diagnosis');
        
        if ($nextDiagnosis.length > 0) {
            // Hide current diagnosis
            $currentDiagnosis.addClass('hidden-diagnosis');
            
            // Show next diagnosis
            $nextDiagnosis.removeClass('hidden-diagnosis');
            
            // Scroll to top of diagnosis
            $('html, body').animate({
                scrollTop: $('.result-content').offset().top - 100
            }, 500);
        } else {
            // No more diagnoses, show completion message with emphasis on View Results
            Swal.fire({
                icon: 'info',
                title: 'All Diagnoses Complete',
                text: 'You have completed all available diagnoses. Would you like to view your results?',
                allowOutsideClick: false,
                allowEscapeKey: false,
                showCancelButton: true,
                confirmButtonText: 'View Results',
                cancelButtonText: 'Stay Here',
                buttonsStyling: false,
                customClass: {
                    confirmButton: 'btn btn-success btn-lg swal2-confirm-primary',
                    cancelButton: 'btn btn-outline-secondary swal2-cancel-secondary'
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = getResultsUrl();
                }
            });
        }
    }
    
    // Auto-scroll to results on page load
    if ($('.result-content').length > 0) {
        setTimeout(function() {
            $('html, body').animate({
                scrollTop: $('.result-content').offset().top - 100
            }, 1000);
        }, 500);
    }
    
});