jQuery(document).ready(function($){
	let currentPage = 1;
	let isLoading = false;
	let hasMoreData = true;
	
	// Function to get records with loading functionality
	function getRecords(loadMore = false) {
		var filter = $('.filterBtn.active').data('filter');
		
		// If no active button, default to 'confirmed'
		if (!filter) {
			filter = 'confirmed';
			// Set the confirmed button as active if none is active
			$('.filterBtn[data-filter="confirmed"]').addClass('active');
		}
		
		// Reset pagination if not loading more
		if (!loadMore) {
			currentPage = 1;
			hasMoreData = true;
		}
		
		// Prevent multiple simultaneous requests
		if (isLoading) return;
		isLoading = true;
		
		// Show loading icon
		if (!loadMore) {
			showLoadingIcon();
		} else {
			showLoadMoreSpinner();
		}
		
		$.ajax({
			url: 'process/confirmed-diagnosis.php',
			type: 'POST',
			dataType: 'json',
			timeout: 30000, // 30 second timeout
			data: {
				filter: filter,
				page: currentPage
			},
			success: function(data) {
				if (data.success) {
					if (loadMore) {
						appendPatients(data.patients);
					} else {
						renderPatients(data.patients, data.message);
					}
					
					// Update pagination info
					hasMoreData = data.pagination && data.pagination.has_more;
					updateLoadMoreButton();
					
					if (loadMore) {
						currentPage++;
					}
				} else {
					if (!loadMore) {
						$('#patientContainer-body').html('<div class="error-state"><div class="error-icon">⚠️</div><h3>Loading Error</h3><p>Error loading patient data</p></div>');
					}
				}
				
				if (!loadMore) {
					hideLoadingIcon();
				} else {
					hideLoadMoreSpinner();
				}
				isLoading = false;
			},
			error: function(xhr, status, error) {
				if (!loadMore) {
					hideLoadingIcon();
				} else {
					hideLoadMoreSpinner();
				}
				isLoading = false;
				
				console.log('Error loading patient data:', status, error);
				console.log('Response:', xhr.responseText);
				
				let errorMessage = '';
				if (status === 'timeout') {
					errorMessage = 'Request timed out. Please check your connection and try again.';
				} else if (status === 'error') {
					errorMessage = 'Network error. Please check your connection and try again.';
				} else {
					errorMessage = 'Error loading patient data. Please try again.';
				}
				
				if (!loadMore) {
					$('#patientContainer-body').html(`
						<div class="error-state">
							<div class="error-icon">⚠️</div>
							<h3>Loading Error</h3>
							<p>${errorMessage}</p>
							<button onclick="getRecords()" class="retry-btn">Try Again</button>
						</div>
					`);
				}
			}
		});
	}
	
	// Function to show loading icon
	function showLoadingIcon() {
		// Create enhanced loading overlay if it doesn't exist
		if ($('#loading-overlay').length === 0) {
			var loadingHTML = `
				<div id="loading-overlay">
					<div class="loading-container">
						<!-- Custom CSS Spinner -->
						<div class="custom-spinner">
							<div class="spinner-ring"></div>
							<div class="spinner-pulse"></div>
						</div>
						
						<!-- Loading Text -->
						<h3>Loading Dashboard</h3>
						<p id="loading-message">Fetching patient data...</p>
						
						<!-- Progress dots -->
						<div class="loading-dots">
							<span class="dot"></span>
							<span class="dot"></span>
							<span class="dot"></span>
						</div>
					</div>
				</div>
			`;
			$('body').append(loadingHTML);
		} else {
			$('#loading-overlay').show();
		}
		
		// Add loading state to the active filter button
		$('.filterBtn.active').addClass('loading-btn').prop('disabled', true);
		
		// Animate loading messages
		setTimeout(() => {
			$('#loading-message').text('Processing filters...');
		}, 1000);
		
		setTimeout(() => {
			$('#loading-message').text('Almost ready...');
		}, 2000);
	}
	
	// Function to show load more spinner
	function showLoadMoreSpinner() {
		$('#load-more-btn').prop('disabled', true).html('<div class="spinner-border spinner-border-sm" role="status"><span class="sr-only">Loading...</span></div> Loading...');
	}
	
	// Function to hide load more spinner
	function hideLoadMoreSpinner() {
		$('#load-more-btn').prop('disabled', false).html('Load More');
	}
	
	// Function to update load more button visibility
	function updateLoadMoreButton() {
		if (hasMoreData) {
			if ($('#load-more-container').length === 0) {
				$('#patientContainer-body').after(`
					<div id="load-more-container" style="text-align: center; margin: 20px 0;">
						<button id="load-more-btn" class="btn btn-primary" style="padding: 10px 30px;">Load More</button>
					</div>
				`);
			}
			$('#load-more-container').show();
		} else {
			$('#load-more-container').hide();
		}
	}
	
	// Function to append new patients to existing list
	function appendPatients(patients) {
		if (!patients || patients.length === 0) {
			hasMoreData = false;
			updateLoadMoreButton();
			return;
		}
		
		let html = '';
		patients.forEach(function(patient) {
			html += renderPatientRow(patient);
		});
		
		$('#patientContainer-body').append(html);
	}
	
	// Function to hide loading icon
	function hideLoadingIcon() {
		// Add fade out animation before hiding
		$('#loading-overlay').css('animation', 'fadeOut 0.3s ease-out');
		setTimeout(() => {
			$('#loading-overlay').hide();
			$('#loading-overlay').css('animation', ''); // Reset animation
		}, 300);
		
		// Remove loading state from buttons
		$('.filterBtn').removeClass('loading-btn').prop('disabled', false);
		
		// Reset loading message for next time
		setTimeout(() => {
			$('#loading-message').text('Fetching patient data...');
		}, 500);
	}
	
	// Bind click events to filter buttons
	$('.filterBtn').on('click', function(e) {
		e.preventDefault();
		
		// Remove active class from all buttons
		$('.filterBtn').removeClass('active');
		
		// Add active class to clicked button
		$(this).addClass('active');
		
		// Reset pagination and load records with the new filter
		currentPage = 1;
		hasMoreData = true;
		$('#load-more-container').remove(); // Remove existing load more button
		getRecords();
	});
	
	// Bind click event to load more button
	$(document).on('click', '#load-more-btn', function(e) {
		e.preventDefault();
		currentPage++;
		getRecords(true);
	});
	
	// Initial load on page ready
	setTimeout(function() {
		getRecords();
	}, 100);
	
	// Export function for manual calling
	window.getRecords = getRecords;
	
	// Function to render patients from JSON data
	function renderPatients(patients, message = '') {
		// Remove existing load more button
		$('#load-more-container').remove();
		
		if (!patients || patients.length === 0) {
			// Use custom message if provided, otherwise use default
			let displayMessage = message || 'No patients in queue at this time';
			$('#patientContainer-body').html(`<div class="empty-state">${displayMessage}</div>`);
			hasMoreData = false;
			return;
		}
		
		let html = '';
		
		patients.forEach(function(patient) {
			html += renderPatientRow(patient);
		});
		
		$('#patientContainer-body').html(html);
		
		// Update load more button
		updateLoadMoreButton();
	}
	
	// Function to render a single patient row
	function renderPatientRow(patient) {
		let diagnosisText = patient.diagnosis;
		if (patient.is_sepsis && patient.acuity >= 5) {
			diagnosisText += " (SEPSIS)";
		} else if (patient.is_sirs) {
			diagnosisText += " (SIRS)";
		}
		
		let physiciansOptions = '<option>Assign Physician</option>';
		patient.physicians.forEach(function(physician) {
			let selected = physician.selected ? 'selected' : '';
			physiciansOptions += `<option ${selected} value="${physician.id}">${physician.name}</option>`;
		});
		
		// Build action buttons
		let actionButtons = generateActionButtons(patient);
		
		return `
			<div class="tr" id="${patient.waitingroom_guid}" data-user-id="${patient.patient_id}">
				<input type="hidden" id="waitingEntryID" value="${patient.waitingroom_id}">
				
				<!-- Name -->
				<div class="td patient_info">
					<strong>${patient.patient_name}</strong>
					<span class="phone">${patient.phone}</span>
					<span class="age">${patient.age}</span>
					<span class="returning">${patient.is_returning_patient ? 'Returning Patient' : 'New Patient'}</span>
				</div>
				
				<!-- Diagnosis -->
				<div class="td patient_symptoms">
					${diagnosisText}
					<strong>${patient.symptoms}</strong>
					${patient.symptom_photo ? `<a href="${patient.symptom_photo}" class="symptom_photo" data-fancybox data-caption="${patient.symptoms}">View Photo</a>` : ''}
				</div>
				
				<!-- Assign To -->
				<div class="td patient_assign">
					<select class="btn btn-primary col-md-12 physician-dropdown" for="${patient.waitingroom_id}">
						${physiciansOptions}
					</select>
					<a href="encounter-notes.php?waiting_room_id=${patient.waitingroom_id}">
						<button class="btn btn-primary col-md-12 view_notes">View Notes</button>
					</a>
				</div>
				
				<!-- Status -->
				<div class="td patient_status ${patient.status_class}" style="text-align: center;">
					<b class="inOutText">${patient.status}</b>
				</div>
				
				<!-- Acuity -->
				<div class="td patient_acuity" style="text-align: center;">
					<input type="hidden" id="acuityValue" value="${patient.acuity}">
					<h3 class="card-subtitle mb-2 ${patient.status_class}">
						<b class="acuityText">${patient.acuity}/10</b>
					</h3>
				</div>
				
				<!-- Vitals -->
				<div class="td patient_vitals">
					<button class="btn btn-primary col-md-12 addVitals" data-toggle="modal" represents="${patient.waitingroom_guid}" data-target="#vitals" style="margin-bottom: 10px;" data-patient-id="${patient.patient_id}">Add</button>
					<a href="patient_vitals.php?patient_id=${patient.patient_id}" class="btn btn-primary addVitals" data-patient-id="${patient.patient_id}">View</a>
				</div>
				
				<!-- Wait Time -->
				<div class="td patient_waitTime">
					<h3 class="card-subtitle mb-2 text-muted">
						<input type="hidden" id="checkinTime" value="${patient.wait_time.checkin_time}">
						<input type="hidden" id="hrs" value="${patient.wait_time.hours}">
						<input type="hidden" id="mins" value="${patient.wait_time.minutes}">
						<b>${patient.wait_time.formatted}</b>
					</h3>
					<h6 class="card-subtitle mb-2 text-muted"><b>DD:HH:MM</b></h6>
				</div>
				
				<!-- Action -->
				<div class="td patient_action">
					<a href="waitingroom_delete.php?wID=${patient.waitingroom_id}&request_from=nurse">
						<button class="btn btn-danger col-md-12" style="margin-bottom: 10px; display: block">Delete</button>
					</a>
					<a href="write-order.php?uid=${patient.user_id}&patient_id=${patient.patient_id}">
						<button class="btn btn-primary col-md-12">Order</button>
					</a>
					${actionButtons}
				</div>
				
				<div class="clear"></div>
			</div>
			<div class="clear"></div>
		`;
	}
	
	// Function to generate action buttons (placeholder for now)
	function generateActionButtons(patient) {
		// These would need to be implemented based on the original functions
		// adt_dashboard_button(), video_call(), call_request()
		return `
			<!-- Additional action buttons can be added here -->
			<!-- adt_dashboard_button, video_call, call_request functions would generate these -->
		`;
	}
});
