// Manage Devices - Provider JavaScript
$(document).ready(function() {
    
    /* ================================================================== */
    // Add Device Button Click
    /* ================================================================== */
    $('.btn-add').on('click', function() {
        var deviceType = $(this).data('device-type');
        var patientId = $('body').data('patient-id');
        var deviceItem = $(this).closest('.device-item');
        
        // Determine device label and ID field name
        var deviceLabel = '';
        var idFieldName = '';
        
        switch(deviceType) {
            case 'anyhub':
                deviceLabel = 'AnyHub (Cellular Hub)';
                idFieldName = 'Serial Number';
                break;
            case 'bp':
                deviceLabel = 'Blood Pressure Cuff';
                idFieldName = 'IMEI Number';
                break;
            case 'weight':
                deviceLabel = 'Weighing Scale';
                idFieldName = 'IMEI Number';
                break;
        }
        
        // Show SweetAlert2 dialog
        Swal.fire({
            title: 'Add ' + deviceLabel,
            html: '<p style="margin-bottom: 10px;">Enter the device ' + idFieldName + ':</p>',
            input: 'text',
            inputPlaceholder: 'Enter ' + idFieldName,
            showCancelButton: true,
            confirmButtonText: 'Add Device',
            confirmButtonColor: '#28a745',
            cancelButtonColor: '#6c757d',
            allowOutsideClick: false,
            inputValidator: function(value) {
                if (!value) {
                    return 'Please enter a valid ' + idFieldName + '!';
                }
                if (value.length < 5) {
                    return idFieldName + ' must be at least 5 characters!';
                }
            }
        }).then(function(result) {
            if (result.isConfirmed) {
                var imeiNumber = result.value;
                
                // Make AJAX call to add device
                $.ajax({
                    type: 'POST',
                    url: 'process/provider-add-device.php',
                    data: {
                        patient_id: patientId,
                        device_type: deviceType,
                        imei_number: imeiNumber
                    },
                    dataType: 'json',
                    beforeSend: function() {
                        Swal.fire({
                            title: 'Adding Device...',
                            text: 'Please wait',
                            allowOutsideClick: false,
                            allowEscapeKey: false,
                            didOpen: function() {
                                Swal.showLoading();
                            }
                        });
                    },
                    success: function(response) {
                        if (response.status_code == '200') {
                            // Success - update UI
                            Swal.fire({
                                icon: 'success',
                                title: 'Success!',
                                text: deviceLabel + ' has been added successfully.',
                                confirmButtonColor: '#28a745'
                            }).then(function() {
                                // Update the device item
                                deviceItem.find('.device-status')
                                    .removeClass('disconnected')
                                    .addClass('connected')
                                    .text('Connected');
                                
                                deviceItem.find('.device-imei').text(imeiNumber);
                                
                                deviceItem.find('.device-actions').html(
                                    '<button class="btn-action btn-remove" data-imei="' + imeiNumber + '" data-device-type="' + deviceType + '">Remove Device</button>'
                                );
                            });
                        } else if (response.status_code == '403') {
                            // Duplicate entry
                            Swal.fire({
                                icon: 'error',
                                title: 'Duplicate Device',
                                html: 'This device is already registered to:<br><strong>' + response.patient_fullname + '</strong>',
                                confirmButtonColor: '#dc3545'
                            });
                        } else {
                            // Other error
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.status_description || 'Failed to add device. Please try again.',
                                confirmButtonColor: '#dc3545'
                            });
                        }
                    },
                    error: function(xhr, status, error) {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: 'An error occurred while adding the device. Please try again.',
                            confirmButtonColor: '#dc3545'
                        });
                        console.error('Error:', error);
                    }
                });
            }
        });
    });
    
    /* ================================================================== */
    // Remove Device Button Click (Event Delegation)
    /* ================================================================== */
    $(document).on('click', '.btn-remove', function() {
        var deviceType = $(this).data('device-type');
        var imeiNumber = $(this).data('imei');
        var patientId = $('body').data('patient-id');
        var deviceItem = $(this).closest('.device-item');
        
        // Determine device label
        var deviceLabel = '';
        switch(deviceType) {
            case 'anyhub':
                deviceLabel = 'AnyHub (Cellular Hub)';
                break;
            case 'bp':
                deviceLabel = 'Blood Pressure Cuff';
                break;
            case 'weight':
                deviceLabel = 'Weighing Scale';
                break;
        }
        
        // Show confirmation dialog
        Swal.fire({
            title: 'Remove ' + deviceLabel + '?',
            html: 'Are you sure you want to remove this device?<br><br><strong>IMEI: ' + imeiNumber + '</strong><br><br>This action cannot be undone.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, remove it!',
            cancelButtonText: 'Cancel'
        }).then(function(result) {
            if (result.isConfirmed) {
                // Make AJAX call to remove device
                $.ajax({
                    type: 'POST',
                    url: 'process/provider-remove-device.php',
                    data: {
                        patient_id: patientId,
                        device_type: deviceType,
                        imei_number: imeiNumber
                    },
                    dataType: 'json',
                    beforeSend: function() {
                        Swal.fire({
                            title: 'Removing Device...',
                            text: 'Please wait',
                            allowOutsideClick: false,
                            allowEscapeKey: false,
                            didOpen: function() {
                                Swal.showLoading();
                            }
                        });
                    },
                    success: function(response) {
                        if (response.status_code == '200') {
                            // Success - update UI
                            Swal.fire({
                                icon: 'success',
                                title: 'Removed!',
                                text: deviceLabel + ' has been removed successfully.',
                                confirmButtonColor: '#28a745'
                            }).then(function() {
                                // Update the device item
                                deviceItem.find('.device-status')
                                    .removeClass('connected')
                                    .addClass('disconnected')
                                    .text('Not Connected');
                                
                                deviceItem.find('.device-imei').text('Not Connected');
                                
                                deviceItem.find('.device-actions').html(
                                    '<button class="btn-action btn-add" data-device-type="' + deviceType + '">Add Device</button>'
                                );
                            });
                        } else {
                            // Error
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.status_description || 'Failed to remove device. Please try again.',
                                confirmButtonColor: '#dc3545'
                            });
                        }
                    },
                    error: function(xhr, status, error) {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: 'An error occurred while removing the device. Please try again.',
                            confirmButtonColor: '#dc3545'
                        });
                        console.error('Error:', error);
                    }
                });
            }
        });
    });
    
    /* ================================================================== */
    // Edit Device Button Click (Future Enhancement)
    /* ================================================================== */
    $(document).on('click', '.btn-edit', function() {
        var deviceType = $(this).data('device-type');
        var currentImei = $(this).data('imei');
        var patientId = $('body').data('patient-id');
        var deviceItem = $(this).closest('.device-item');
        
        Swal.fire({
            title: 'Edit Device',
            text: 'This feature is coming soon!',
            icon: 'info',
            confirmButtonColor: '#007bff'
        });
    });
    
});
