$(document).ready(function(){
    
    // Handle patient creation form submission
    $('#new_patient_form').on('submit', function(e) {
        e.preventDefault(); // Prevent default form submission
        
        // Check for validation errors before submitting
        if($('.validation-error').length > 0 || 
           $('input[name="username"]').hasClass('is-invalid') || 
           $('input[name="patient_email"]').hasClass('is-invalid') ||
           $('input[name="mobile_phone"]').hasClass('is-invalid')) {
            
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fix the validation errors before submitting the form.',
                confirmButtonText: 'OK'
            });
            return false;
        }
        
        // Combine country code and phone number before submitting
        var countryCode = $('#country_code').val();
        var phoneNumber = $('#mobile_phone').val().trim();
        
        if (phoneNumber) {
            // Remove any spaces, hyphens, parentheses from phone number
            phoneNumber = phoneNumber.replace(/[\s\-\(\)\.]/g, '');
            // Combine country code with phone number
            $('#mobile_phone').val(countryCode + phoneNumber);
        }
        
        // Get form data
        var formData = $(this).serialize();
        
        // Disable submit button and show loading
        var submitBtn = $(this).find('button[type="submit"]');
        var originalText = submitBtn.text();
        submitBtn.prop('disabled', true).text('Processing...');
        
        // Show loading alert
        Swal.fire({
            title: 'Processing...',
            text: 'Adding new patient, please wait.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            showConfirmButton: false,
            didOpen: function() {
                Swal.showLoading();
            }
        });
        
        // Submit form via Ajax
        $.ajax({
            url: 'process/add_new_patient.php',
            type: 'POST',
            data: formData,
            dataType: 'json',
            success: function(response) {
                // Re-enable submit button
                submitBtn.prop('disabled', false).text(originalText);
                
                // Close loading alert
                Swal.close();
                
                if(response.success === true) {
                    // Success - show success message then redirect
                    Swal.fire({
                        icon: 'success',
                        title: 'Success!',
                        text: response.message,
                        confirmButtonText: 'OK',
                        allowOutsideClick: false
                    }).then(function(result) {
                        if (result.isConfirmed) {
                            window.location.href = 'new_records.php';
                        }
                    });
                } else {
                    // Error - show error message
                    Swal.fire({
                        icon: 'error',
                        title: 'Error!',
                        text: response.message,
                        confirmButtonText: 'OK'
                    });
                }
            },
            error: function(xhr, status, error) {
                // Re-enable submit button
                submitBtn.prop('disabled', false).text(originalText);
                
                // Close loading alert
                Swal.close();
                
                // Show error message
                Swal.fire({
                    icon: 'error',
                    title: 'Connection Error',
                    text: 'There was an error connecting to the server. Please try again.',
                    confirmButtonText: 'OK'
                });
                
                console.error('Ajax Error:', error);
                console.error('Status:', status);
                console.error('Response:', xhr.responseText);
            }
        });
    });
    
    // Form validation - check if username exists
    $('#new_patient_form input[name="username"]').on('blur', function() {
        var username = $(this).val();
        var inputField = $(this);
        
        // Remove previous error
        inputField.removeClass('is-invalid');
        inputField.next('.validation-error').remove();
        
        if(username.length > 0) {
            $.ajax({
                url: 'process/check_username_availability.php',
                type: 'POST',
                data: { username: username },
                dataType: 'json',
                success: function(response) {
                    if(response.exists) {
                        inputField.addClass('is-invalid');
                        inputField.after('<span class="validation-error">Username already exists</span>');
                    }
                }
            });
        }
    });
    
    // Form validation - check if email exists (only if email is provided)
    $('#new_patient_form input[name="patient_email"]').on('blur', function() {
        var email = $(this).val();
        var inputField = $(this);
        
        // Remove previous error
        inputField.removeClass('is-invalid');
        inputField.next('.validation-error').remove();
        
        if(email.length > 0) {
            // Validate email format first
            if(!isValidEmail(email)) {
                inputField.addClass('is-invalid');
                inputField.after('<span class="validation-error">Please enter a valid email address</span>');
                return;
            }
            
            $.ajax({
                url: 'process/check_email_availability.php',
                type: 'POST',
                data: { email: email },
                dataType: 'json',
                success: function(response) {
                    if(response.exists) {
                        inputField.addClass('is-invalid');
                        inputField.after('<span class="validation-error">Email already exists</span>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Email check error:', error);
                }
            });
        }
    });
    
    // Form validation - check if phone exists (only if phone is provided)
    $('#new_patient_form input[name="mobile_phone"]').on('blur', function() {
        var phone = $(this).val();
        var inputField = $(this);
        
        // Remove previous error
        inputField.removeClass('is-invalid');
        inputField.next('.validation-error').remove();
        
        if(phone.length > 0) {
            $.ajax({
                url: 'process/check_phone_availability.php',
                type: 'POST',
                data: { phone: phone },
                dataType: 'json',
                success: function(response) {
                    if(response.exists) {
                        inputField.addClass('is-invalid');
                        inputField.after('<span class="validation-error">Phone number already exists</span>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Phone check error:', error);
                }
            });
        }
    });
    
    // Helper function to validate email format
    function isValidEmail(email) {
        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    // Password strength indicator (optional enhancement)
    $('#new_patient_form input[name="password"]').on('keyup', function() {
        var password = $(this).val();
        var strength = 0;
        
        if(password.length >= 8) strength++;
        if(password.match(/[a-z]+/)) strength++;
        if(password.match(/[A-Z]+/)) strength++;
        if(password.match(/[0-9]+/)) strength++;
        if(password.match(/[$@#&!]+/)) strength++;
        
        // You can add a strength indicator UI here if desired
    });
    
});
