<?php
    session_start();

    // Sleep for 10 seconds to simulate processing delay
    // sleep(10);

    // Set content type to JSON first, before any output
    header('Content-Type: application/json');
    
    $sessionUserID = $_SESSION['userid'];
    $userType = $_SESSION['userType'];
    
    // Redirect if not logged in properly
    if(!isset($sessionUserID) || !isset($userType)) {
        echo json_encode(array('success' => false, 'error' => 'Authentication required'));
        die();
    }

    // Include functions and database connection
    include('functions.php');
    
    // Include the caching system
    include('SimpleCache.php');
    
    global $callarray;
    $callarray = array();
    
    // Initialize cache system with 5 minute TTL
    try {
        $cache = new SimpleCache(null, 300); // 5 minutes cache
    } catch (Exception $e) {
        // If cache fails to initialize, continue without caching
        $cache = null;
        error_log("Cache initialization failed: " . $e->getMessage());
    }
    
    // Get user id from patient ID with 7-day caching
    function get_user_id($patient_id, $cache = null){
        if ($cache) {
            $cache_key = "user_id_" . $patient_id;
            $cached_result = $cache->get($cache_key);
            if ($cached_result !== null) {
                return $cached_result;
            }
        }
        
        $sql = "SELECT `all_users`.`userid` FROM `all_users` INNER JOIN `patients` ON `all_users`.`userid` = `patients`.`user_id` WHERE `patients`.`patient_id` = '$patient_id'";
        $result = ddxrxDBConnect()->query($sql);
        $row = $result->fetch_assoc();
        $user_id = isset($row['userid']) ? $row['userid'] : null;
        
        // Cache for 7 days (604800 seconds)
        if ($cache && $user_id !== null) {
            $cache->set($cache_key, $user_id, 604800);
        }
        
        return $user_id;
    }
    
    // Get physicians list with 7-day caching
    function get_physicians_list($version, $cache = null) {
        if ($cache) {
            $cache_key = "physicians_list_" . $version;
            $cached_result = $cache->get($cache_key);
            if ($cached_result !== null) {
                return $cached_result;
            }
        }
        
        $physicians_list = array();
        $connection = ddxrxDBConnect();
        $users = "SELECT physician.physician_fname, physician.physician_lname, physician.id FROM all_users join physician on physician.user_id = all_users.userid where user_type = 2 and location_id =1 and physician.groupid = '$version'";
        $users_query = mysqli_query($connection, $users);
        
        while($row = mysqli_fetch_array($users_query)) {
            $physicians_list[] = array(
                'id' => $row['id'],
                'name' => "Dr. ".$row['physician_fname']." ".$row['physician_lname']
            );
        }
        
        // Cache for 7 days (604800 seconds)
        if ($cache) {
            $cache->set($cache_key, $physicians_list, 604800);
        }
        
        return $physicians_list;
    }
    
    // Get vitals data with 24-hour caching
    function get_patient_vitals($patient_ids, $cache = null) {
        if (empty($patient_ids)) {
            return array();
        }
        
        $vitals_data = array();
        
        // Check cache for each patient's vitals
        $uncached_patients = array();
        if ($cache) {
            foreach ($patient_ids as $patient_id) {
                $cache_key = "patient_vitals_" . $patient_id;
                $cached_vitals = $cache->get($cache_key);
                if ($cached_vitals !== null) {
                    $vitals_data[$patient_id] = $cached_vitals;
                } else {
                    $uncached_patients[] = $patient_id;
                }
            }
        } else {
            $uncached_patients = $patient_ids;
        }
        
        // Query uncached patients
        if (!empty($uncached_patients)) {
            $connection = ddxrxDBConnect();
            $patient_ids_str = "'" . implode("','", array_map('mysqli_real_escape_string', array_fill(0, count($uncached_patients), $connection), $uncached_patients)) . "'";
            
            $vitals_query = "SELECT vh.patient_id, vh.systolic, vh.diastolic, vh.heartrate, vh.respiratory, vh.pulseox, vh.temp, vh.glucose 
                           FROM (SELECT `patient_id`, MAX(date_added) AS `MaxDate` FROM `vitals_history` WHERE `patient_id` IN ($patient_ids_str) GROUP BY `patient_id`) AS `max_dates` 
                           LEFT JOIN `vitals_history` vh 
                               ON max_dates.`patient_id` = vh.`patient_id` AND max_dates.`MaxDate` = vh.`date_added`";
                               
            $vitals_result = mysqli_query($connection, $vitals_query);
            
            if ($vitals_result) {
                while ($vitals_row = mysqli_fetch_assoc($vitals_result)) {
                    $patient_id = $vitals_row['patient_id'];
                    $patient_vitals = array(
                        'systolic' => $vitals_row['systolic'],
                        'diastolic' => $vitals_row['diastolic'],
                        'heartrate' => $vitals_row['heartrate'],
                        'respiratory' => $vitals_row['respiratory'],
                        'pulseox' => $vitals_row['pulseox'],
                        'temp' => $vitals_row['temp'],
                        'glucose' => $vitals_row['glucose']
                    );
                    
                    $vitals_data[$patient_id] = $patient_vitals;
                    
                    // Cache for 24 hours (86400 seconds)
                    if ($cache) {
                        $cache_key = "patient_vitals_" . $patient_id;
                        $cache->set($cache_key, $patient_vitals, 86400);
                    }
                }
            }
            
            // Set empty vitals for patients with no vitals data
            foreach ($uncached_patients as $patient_id) {
                if (!isset($vitals_data[$patient_id])) {
                    $empty_vitals = array(
                        'systolic' => null,
                        'diastolic' => null,
                        'heartrate' => null,
                        'respiratory' => null,
                        'pulseox' => null,
                        'temp' => null,
                        'glucose' => null
                    );
                    $vitals_data[$patient_id] = $empty_vitals;
                    
                    // Cache empty vitals for 24 hours too
                    if ($cache) {
                        $cache_key = "patient_vitals_" . $patient_id;
                        $cache->set($cache_key, $empty_vitals, 86400);
                    }
                }
            }
        }
        
        return $vitals_data;
    }
    
    $version = get_version();
    
    $vitals = array();
    $vitals[] = array('fieldName' => 'systolic', 'fieldLable' => 'Systolic BP', 'min' => 90, 'max' => 200);
    $vitals[] = array('fieldName' => 'diastolic', 'fieldLable' => 'Diastolic BP', 'min' => 50, 'max' => 100);
    $vitals[] = array('fieldName' => 'heartrate', 'fieldLable' => 'Heart Rate', 'min' => 50, 'max' => 100);
    $vitals[] = array('fieldName' => 'respiratory', 'fieldLable' => 'Respiratory Rate', 'min' => 10, 'max' => 22);
    $vitals[] = array('fieldName' => 'pulseox', 'fieldLable' => 'Pulse Ox', 'min' => 88, 'max' => 100);
    $vitals[] = array('fieldName' => 'temp', 'fieldLable' => 'Temperature', 'min' => 95, 'max' => 99);
    $vitals[] = array('fieldName' => 'glucose', 'fieldLable' => 'Temperature', 'min' => 90, 'max' => 300);
    
    // Get filter from POST data
    $filter = isset($_POST['filter']) ? $_POST['filter'] : 'confirmed'; // Default to 'confirmed' to match active button
    
    // Get pagination parameters (optional)
    $page = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
    $limit = 5; // Limit initial load to 5 entries for better performance
    $offset = ($page - 1) * $limit;
    
    // Set time interval based on user type
    $time_interval = ($userType == 'doctor') ? '48 HOUR' : '7 DAY';
    
    // Create cache key based on user type, filter, version, time interval, and page
    $version = get_version();
    $cache_key = "confirmed_diagnosis_" . $userType . "_" . $filter . "_" . $version . "_" . $time_interval . "_page_" . $page;
    
    // Try to get data from cache first
    $cached_result = null;
    if ($cache) {
        $cached_result = $cache->get($cache_key);
    }
    
    if ($cached_result !== null) {
        // Return cached data
        echo json_encode($cached_result);
        exit;
    }
    
    // Build filter condition
    $filter_condition = "";
    if($filter == 'confirmed') {
        $filter_condition = " AND wr.`chief_complaints` IS NOT NULL";
    } elseif($filter == 'waiting') {
        $filter_condition = " AND wr.`chief_complaints` IS NULL";
    }
    // For 'all' filter, no additional condition needed
    
    // Optimized query: Join patient details only (vitals separated for caching)
    $query = "SELECT wr.*, 
                     p.patient_dob, p.cell
              FROM `waiting_room` wr 
              LEFT JOIN `patients` p ON wr.`patient_id` = p.`patient_id`
              WHERE wr.`checkin_time` >= (NOW() - INTERVAL $time_interval) AND wr.`groupid` = '$version'" 
              . $filter_condition . " 
              ORDER BY wr.`checkin_time` DESC
              LIMIT $limit OFFSET $offset";

    $patients = array();
    $connection = ddxrxDBConnect();
    
    // Get cached physicians list (7-day cache)
    $physicians_list = get_physicians_list($version, $cache);
    
    $result = mysqli_query($connection, $query);
    
    // Check for query errors
    if (!$result) {
        error_log("Query error in confirmed-diagnosis.php: " . mysqli_error($connection));
        echo json_encode(array('success' => false, 'error' => 'Database query failed', 'patients' => array(), 'count' => 0));
        mysqli_close($connection);
        exit;
    }
    
    // First pass: collect all patient data and patient IDs
    $patient_data = array();
    $patient_ids = array();
    
    if (mysqli_num_rows($result) > 0) {
        while($data = mysqli_fetch_assoc($result)) {
            $patient_data[] = $data;
            $patient_ids[] = $data['patient_id'];
        }
    }
    
    // Get vitals data for all patients with 24-hour caching
    $vitals_data = get_patient_vitals($patient_ids, $cache);
    
    // Second pass: process each patient with cached vitals data
    foreach ($patient_data as $data) {
            
            // Use patient details from the joined query (no separate query needed)
            $pid = $data['patient_id'];
            $DOB = $data['patient_dob'];
            $phoneNumber = isset($data['cell']) ? $data['cell'] : '';
            
            // Get vitals for this patient from cached data
            $patient_vitals = isset($vitals_data[$pid]) ? $vitals_data[$pid] : array(
                'systolic' => null,
                'diastolic' => null,
                'heartrate' => null,
                'respiratory' => null,
                'pulseox' => null,
                'temp' => null,
                'glucose' => null
            );
            
            // Calculate age efficiently in PHP
            if ($DOB && $DOB != '0000-00-00') {
                $today = date("Y-m-d");
                $dob_date = date_create($DOB);
                $today_date = date_create($today);
                if ($dob_date && $today_date) {
                    $diff = date_diff($dob_date, $today_date);
                    $age = $diff->format('%y');
                } else {
                    $age = 'N/A';
                }
            } else {
                $age = 'N/A';
            }
            
            // Calculate acuity using nurse logic for both user types
            $diagnosis = $data['diagnosis'];
            $acuity = 0;
            $is_SEPSIS = 0;

            if (!empty($diagnosis)) {
                $acuity = 2;

                $sepsis_diseases = array("pneumonia","urinary tract infection (UTI)", "bacteraemia", "meningitis", "influenza", "bronchitis", "asthma", "chronic obstructive pulmonary disease", "appendicitis", "diverticulitis", "cellulitis", "orchitis", "pelvic inflammatory disease", "Pyelonephritis","Viral Syndrome","sinusitis","HIV","Ebola","Infectious Mononucleosis","Dengue Fever","Strep Throat","Pharyngitis","Cerebral abscess","Endocarditis","Osteomyelitis","Diverticulitis","Tuberculosis","Lyme Disease","Fungal Infection","Billiary tract infection","C. Difficile Colitis","Liver abscess","Sepsis","Pelvic Inflammatory Disease (PID)","Typhoid Fever","Septic Arthritis");

                foreach ($sepsis_diseases as $key => $item) {
                    if (stripos($diagnosis, $item) !== false) {
                        $is_SEPSIS = 1;
                        $acuity = 3;
                    }
                }

                if(strpos($diagnosis,"Myocardial Ischemia") || strpos($diagnosis,"Seizure") || strpos($diagnosis,"Cerebrovascular accident")) {
                    $acuity = 3;
                }
                    
                if(strpos($diagnosis,"Appendicitis") || strpos($diagnosis,"Cholecystitis") || strpos($diagnosis,"Subaracnoid Hemohhrage")
                    || strpos($diagnosis,"Cerebrovascular accident (CVA)") || strpos($diagnosis,"Meningitis" )) {
                    $acuity = 4;
                }
                for ($x = 0; $x < count($vitals); $x++) {
                    $vitalField = $vitals[$x]['fieldName'];
                    if (isset($patient_vitals[$vitalField]) && $patient_vitals[$vitalField] != 0) {
                        if ($patient_vitals[$vitalField] < $vitals[$x]['min'] || $patient_vitals[$vitalField] > $vitals[$x]['max']) {
                            $acuity++;
                        }
                    }
                }                                
            }

            // Calculate status
            if($acuity != "" && ((int) $acuity >= 4)) {
                $inOut = "Inpatient";
                if($is_SEPSIS == 1) $inOut .= " (Sepsis)";
                $statusClass = "text-danger";
            } else if($acuity != "" && ((int) $acuity == 3)) {
                $inOut = "Undecided";
                $statusClass = "text-muted";
            } else {
                $inOut = "Outpatient";
                $statusClass = "text-success";
            }
            
            // Calculate wait time efficiently in PHP instead of separate SQL query
            $rawtime = $data['checkin_time'];
            $checkin_timestamp = strtotime($rawtime);
            $current_timestamp = time();
            
            // Handle invalid dates gracefully
            if ($checkin_timestamp === false || $checkin_timestamp > $current_timestamp) {
                $dayValue = $hourValue = $minuteValue = 0;
            } else {
                $time_diff = $current_timestamp - $checkin_timestamp;
                
                // Calculate days, hours, minutes
                $dayValue = max(0, floor($time_diff / 86400));
                $hourValue = max(0, floor(($time_diff % 86400) / 3600));
                $minuteValue = max(0, floor(($time_diff % 3600) / 60));
            }
            
            $days = str_pad($dayValue, 2, '0', STR_PAD_LEFT);
            $hours = str_pad($hourValue, 2, '0', STR_PAD_LEFT);
            $minutes = str_pad($minuteValue, 2, '0', STR_PAD_LEFT);
            
            // Use pre-fetched physicians list and mark selected physician
            $physicians = array();
            foreach ($physicians_list as $physician) {
                $physicians[] = array(
                    'id' => $physician['id'],
                    'name' => $physician['name'],
                    'selected' => ($data['physician_id'] != 0 && $data['physician_id'] == $physician['id'])
                );
            }
            
            // Build patient object
            $patient = array(
                'waitingroom_guid' => isset($data['waitingroom_guid']) ? $data['waitingroom_guid'] : '',
                'waitingroom_id' => isset($data['waitingroom_id']) ? $data['waitingroom_id'] : '',
                'patient_id' => isset($data['patient_id']) ? $data['patient_id'] : '',
                'patient_name' => ucwords(isset($data['patient_name']) ? $data['patient_name'] : ''),
                'phone' => $phoneNumber,
                'age' => $age . ' yrs',
                'is_returning_patient' => ($data['is_returning_patient'] == 1),
                'diagnosis' => $data['diagnosis'] != "" ? $data['diagnosis'] : ($data['symptoms'] == "" ? "Awaiting chief complaint" : "No diagnosis matched"),
                'symptoms' => $data['symptoms'] != "" ? $data['symptoms'] : "Awaiting symptoms",
                'symptom_photo' => isset($data['symptom_photo']) ? $data['symptom_photo'] : '',
                'is_sepsis' => $is_SEPSIS,
                'is_sirs' => isset($data['is_SIRS']) ? $data['is_SIRS'] : 0,
                'acuity' => $acuity != "" ? $acuity : 0,
                'status' => $inOut,
                'status_class' => $statusClass,
                'wait_time' => array(
                    'days' => $days,
                    'hours' => $hours,
                    'minutes' => $minutes,
                    'formatted' => $days . " : " . $hours . " : " . $minutes,
                    'checkin_time' => isset($data['checkin_time']) ? $data['checkin_time'] : ''
                ),
                'physicians' => $physicians,
                'assigned_physician_id' => isset($data['physician_id']) ? $data['physician_id'] : null,
                'user_id' => get_user_id($data['patient_id'], $cache),
                'vitals' => $patient_vitals
            );
            
            $patients[] = $patient;
        }
    // Close database connection
    mysqli_close($connection);
    
    // Determine appropriate message for empty results
    $message = '';
    if (count($patients) === 0) {
        switch($filter) {
            case 'confirmed':
                $message = 'No patients with confirmed diagnoses at this time';
                break;
            case 'waiting':
                $message = 'No patients awaiting symptoms at this time';
                break;
            case 'all':
                $message = 'No patients in queue at this time';
                break;
            default:
                $message = 'No patients found for the selected filter';
        }
    }
    
    // Prepare response
    $response = array(
        'success' => true,
        'patients' => $patients,
        'count' => count($patients),
        'filter' => $filter,
        'user_type' => $userType,
        'message' => $message,
        'pagination' => array(
            'page' => $page,
            'limit' => $limit,
            'has_more' => count($patients) === $limit // Indicates if there might be more records
        )
    );
    
    // Cache the result if cache is available
    if ($cache) {
        $cache->set($cache_key, $response);
    }
    
    // Return JSON response
    echo json_encode($response);
