$(document).ready(function(){
    
    // Handle nurse creation form submission
    $('#nurse_create_form').on('submit', function(e) {
        e.preventDefault(); // Prevent default form submission
        
        // Check for validation errors before submitting
        if($('.username-error').length > 0 || $('.email-error').length > 0 || 
           $('input[name="username"]').hasClass('is-invalid') || 
           $('input[name="nurse_email"]').hasClass('is-invalid')) {
            
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fix the validation errors before submitting the form.',
                showConfirmButton: true,
                confirmButtonText: 'OK'
            });
            return false;
        }
        
        // Get form data
        var formData = new FormData(this);
        
        // Show loading alert
        Swal.fire({
            title: 'Processing...',
            text: 'Adding new nurse, please wait.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            showConfirmButton: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        // Submit form via Ajax
        $.ajax({
            url: 'process/add_nurse.php',
            type: 'POST',
            data: formData,
            contentType: false,
            processData: false,
            dataType: 'json',
            success: function(response) {
                // Close loading alert
                Swal.close();
                
                if(response.result === true) {
                    // Success - show success message then redirect
                    Swal.fire({
                        icon: response.icon,
                        title: response.title,
                        text: response.message,
                        showConfirmButton: true,
                        confirmButtonText: 'OK'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            // Redirect to nurses list page
                            window.location.href = 'nurses.php';
                        }
                    });
                } else {
                    // Error - show error message
                    Swal.fire({
                        icon: response.icon,
                        title: response.title,
                        text: response.message,
                        showConfirmButton: true,
                        confirmButtonText: 'OK'
                    });
                }
            },
            error: function(xhr, status, error) {
                // Close loading alert
                Swal.close();
                
                // Show error message
                Swal.fire({
                    icon: 'error',
                    title: 'Connection Error',
                    text: 'There was an error connecting to the server. Please try again.',
                    showConfirmButton: true,
                    confirmButtonText: 'OK'
                });
                
                console.error('Ajax Error:', error);
                console.error('Status:', status);
                console.error('Response:', xhr.responseText);
            }
        });
    });
    
    // Form validation - check if username exists
    $('input[name="username"]').on('blur', function() {
        var username = $(this).val();
        
        if(username.length > 0) {
            $.ajax({
                url: 'process/check_username.php',
                type: 'POST',
                data: { username: username },
                dataType: 'json',
                success: function(response) {
                    if(response.exists) {
                        $('input[name="username"]').addClass('is-invalid');
                        if(!$('.username-error').length) {
                            $('input[name="username"]').after('<div class="username-error text-danger">Username already exists</div>');
                        }
                    } else {
                        $('input[name="username"]').removeClass('is-invalid');
                        $('.username-error').remove();
                    }
                }
            });
        }
    });
    
    // Form validation - check if email exists
    $('input[name="nurse_email"]').on('blur', function() {
        var email = $(this).val();
        
        if(email.length > 0 && isValidEmail(email)) {
            $.ajax({
                url: 'process/check_email.php',
                type: 'POST',
                data: { email: email },
                dataType: 'json',
                success: function(response) {
                    console.log('Email check response:', response); // Debug log
                    
                    if(response.error) {
                        console.error('Email check error:', response.error);
                        // Don't show error to user, just log it
                        $('input[name="nurse_email"]').removeClass('is-invalid');
                        $('.email-error').remove();
                    } else if(response.exists === true) {
                        $('input[name="nurse_email"]').addClass('is-invalid');
                        if(!$('.email-error').length) {
                            $('input[name="nurse_email"]').after('<div class="email-error text-danger">Email already exists</div>');
                        }
                    } else {
                        $('input[name="nurse_email"]').removeClass('is-invalid');
                        $('.email-error').remove();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Ajax error:', error);
                    console.error('Response text:', xhr.responseText);
                    // Remove any existing error styling on connection error
                    $('input[name="nurse_email"]').removeClass('is-invalid');
                    $('.email-error').remove();
                }
            });
        } else if(email.length > 0 && !isValidEmail(email)) {
            // Invalid email format
            $('input[name="nurse_email"]').addClass('is-invalid');
            if(!$('.email-error').length) {
                $('input[name="nurse_email"]').after('<div class="email-error text-danger">Please enter a valid email address</div>');
            }
        } else {
            // Empty email - remove validation styling
            $('input[name="nurse_email"]').removeClass('is-invalid');
            $('.email-error').remove();
        }
    });
    
    // Helper function to validate email format
    function isValidEmail(email) {
        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    // Handle edit nurse form submission
    $('#edit_nurse_form').on('submit', function(e) {
        e.preventDefault(); // Prevent default form submission
        
        // Check for validation errors before submitting
        if($('.email-error').length > 0 || 
           $('input[name="nurse_email"]').hasClass('is-invalid')) {
            
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fix the validation errors before submitting the form.',
                showConfirmButton: true,
                confirmButtonText: 'OK'
            });
            return false;
        }
        
        // Get form data
        var formData = new FormData(this);
        
        // Show loading alert
        Swal.fire({
            title: 'Processing...',
            text: 'Updating nurse information, please wait.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            showConfirmButton: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        // Submit form via Ajax
        $.ajax({
            url: 'process/edit_nurse.php',
            type: 'POST',
            data: formData,
            contentType: false,
            processData: false,
            dataType: 'json',
            success: function(response) {
                // Close loading alert
                Swal.close();
                
                console.log('Edit nurse response:', response); // Debug log
                
                if(response.result === true) {
                    // Success - show success message then redirect
                    Swal.fire({
                        icon: response.icon,
                        title: response.title,
                        text: response.message,
                        showConfirmButton: true,
                        confirmButtonText: 'OK'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            // Redirect to nurses list page
                            window.location.href = 'nurses.php';
                        }
                    });
                } else {
                    // Error - show error message
                    var errorMessage = response.message;
                    if(response.debug_error) {
                        console.error('MySQL Error:', response.debug_error);
                        errorMessage += ' (Check console for details)';
                    }
                    
                    Swal.fire({
                        icon: response.icon,
                        title: response.title,
                        text: errorMessage,
                        showConfirmButton: true,
                        confirmButtonText: 'OK'
                    });
                }
            },
            error: function(xhr, status, error) {
                // Close loading alert
                Swal.close();
                
                // Show error message
                Swal.fire({
                    icon: 'error',
                    title: 'Connection Error',
                    text: 'There was an error connecting to the server. Please try again.',
                    showConfirmButton: true,
                    confirmButtonText: 'OK'
                });
                
                console.error('Ajax Error:', error);
                console.error('Status:', status);
                console.error('Response:', xhr.responseText);
            }
        });
    });
    
    // Form validation for edit nurse - check if email exists (excluding current nurse)
    $('#edit_nurse_form input[name="nurse_email"]').on('blur', function() {
        var email = $(this).val();
        var nurse_id = $('input[name="nurse_id"]').val();
        
        if(email.length > 0 && isValidEmail(email) && nurse_id) {
            $.ajax({
                url: 'process/check_email_edit.php',
                type: 'POST',
                data: { 
                    email: email,
                    nurse_id: nurse_id 
                },
                dataType: 'json',
                success: function(response) {
                    console.log('Email edit check response:', response); // Debug log
                    
                    if(response.error) {
                        console.error('Email check error:', response.error);
                        // Don't show error to user, just log it
                        $('#edit_nurse_form input[name="nurse_email"]').removeClass('is-invalid');
                        $('#edit_nurse_form .email-error').remove();
                    } else if(response.exists === true) {
                        $('#edit_nurse_form input[name="nurse_email"]').addClass('is-invalid');
                        if(!$('#edit_nurse_form .email-error').length) {
                            $('#edit_nurse_form input[name="nurse_email"]').after('<div class="email-error text-danger">Email already exists</div>');
                        }
                    } else {
                        $('#edit_nurse_form input[name="nurse_email"]').removeClass('is-invalid');
                        $('#edit_nurse_form .email-error').remove();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Ajax error:', error);
                    console.error('Response text:', xhr.responseText);
                    // Remove any existing error styling on connection error
                    $('#edit_nurse_form input[name="nurse_email"]').removeClass('is-invalid');
                    $('#edit_nurse_form .email-error').remove();
                }
            });
        } else if(email.length > 0 && !isValidEmail(email)) {
            // Invalid email format
            $('#edit_nurse_form input[name="nurse_email"]').addClass('is-invalid');
            if(!$('#edit_nurse_form .email-error').length) {
                $('#edit_nurse_form input[name="nurse_email"]').after('<div class="email-error text-danger">Please enter a valid email address</div>');
            }
        } else {
            // Empty email or no nurse_id - remove validation styling
            $('#edit_nurse_form input[name="nurse_email"]').removeClass('is-invalid');
            $('#edit_nurse_form .email-error').remove();
        }
    });

});